/*
This file is part of LandscapeTools, version 2.3.

Copyright (c) 2006-2019, Instituto de Tecnologia Quimica e Biologica,
Universidade Nova de Lisboa, Portugal.

LandscapeTools is free software: you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation, either version 2 of the License, or (at your
option) any later version.

LandscapeTools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with LandscapeTools.  If not, see <http://www.gnu.org/licenses/>.

For further details and info check the README file.

You can get LandscapeTools at www.itqb.unl.pt/simulation
*/


/**************************************************************************

GETDENSITY: This program generates probability density maps that can be
  read and interpreted with GETBASINS. It can also be used standalone,
  working as a kernel density estimator. See README for more details.

**************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <math.h>
#include <getopt.h>
#include <unistd.h>
#include <limits.h>


#define PACKAGE "LandscapeTools 2.3"
#define MAXSTR 500
/*#define KERNEL_CUTOFF 1e-5*/
#define GRID_BORDER 1
#define MAX_ENERGY 1e6
#define min(x,y) ((x)<(y) ? (x) : (y))
#define max(x,y) ((x)>(y) ? (x) : (y))

typedef float real ;
typedef char string[MAXSTR] ;
typedef enum {E, P} type;
typedef enum {gauss, triang, naive} kernel_type;
typedef struct {
  int bin ;
  real density, *r ;
} point_t ;
typedef struct {
  int nelem, *elem ;  
  real density, *r ;
} bin_t ;

double erf(double x);
void parse_arguments(int argc, char **argv) ;
void read_data(void) ;
void h_rule_of_thumb(void);
void make_grid(void) ;
void single2multi(int k, int **iv) ;
int multi2single(int *iv) ;
void compute_density(real kr) ;
void compute_density_adaptive(real kr) ;
void compute_density_max(void) ;
void get_neighbors(int d, int *low, int *high, int **neig, int aux[]) ;
real (*kernel)(real *rv) ;
/*real kernel(real *rv) ;*/
real get_kernel_range(int dim);
real kernel_range(void) ;
real kernel_gauss(real *rv);
real kernel_triang(real *rv);
real kernel_naive(real *rv);
void write_fld(type ftype) ;
void write_binary(FILE *f_fld , int dim, type ftype, int iv[]);
void write_2D_gnuplot(type ftype) ;
void write_point_data(void);
real invexp(real x) ;
int ndigits(int i) ;
void message(char mtype, char *format, ...) ;


real sensitivity, dr, h=0, density_max, *rmin, *rmax ;
int ndim, adaptive, w_pdat=0, w_E=0 ;
int npoints, nbins, *gsize ;   /* change to long */
string rname, data_file ;
kernel_type krnl=triang;
bin_t *bin ;
point_t *point ;
FILE *fp_log ;

int main(int argc, char **argv)
{
  real kr;

  parse_arguments(argc, argv) ;
  fp_log = stdout ;
  fprintf(fp_log, "*** GETDENSITY (part of %s) ***\n", PACKAGE) ;
  fprintf(fp_log, "DIM = %d\n", ndim) ;

  rmin = calloc(ndim, sizeof(real));
  rmax = calloc(ndim, sizeof(real));
  gsize = calloc(ndim, sizeof(int));


  read_data() ;

  if (! h) 
    {
      h_rule_of_thumb();
      fprintf(fp_log, "Please check automatic value of h. You may decide to repeat the analysis with a subjective value using option -h.\nh = %g\n", h) ;
    }

  make_grid() ;

  /*kr=kernel_range();*/

  switch (krnl)
    {
    case gauss:
      fprintf(fp_log, "Kernel: Gaussian\n") ;
      kr = get_kernel_range(ndim);
      kernel = kernel_gauss ;
      break;
    case triang:
      fprintf(fp_log, "Kernel: Triangular\n") ;
      if (adaptive)
	message('E', "Sensitivity only applies to Gaussian kernel.\n");
      kr = 1;
      kernel = kernel_triang ;
      break;
    case naive:
      fprintf(fp_log, "Kernel: Naive or rectangular\n") ;
      if (adaptive)
	message('E', "Sensitivity only applies to Gaussian kernel.\n");
      kr = 1;
      kernel = kernel_naive ;
      break;
    default:
      message('E', "Kernel definition.\n");
    }
     
  fprintf(fp_log, "Kernel range = %g\n", kr) ;

  compute_density(kr) ;
  if (adaptive) compute_density_adaptive(kr) ;
  compute_density_max() ;
  
  write_fld(P) ;
  if (ndim <= 2) write_2D_gnuplot(P);
  if (w_E) 
    {
      write_fld(E);
      if (ndim <= 2) write_2D_gnuplot(E);
    }
  free(bin);
  if (w_pdat) write_point_data();


  fprintf(fp_log, "Program normal termination.\n") ;
  return 0 ;
}


/*Reads arguments*/
void parse_arguments(int argc, char **argv)
{
  int c;

 
  while ((c = getopt(argc, argv, "h:s:k:FE")) != -1)
    {
      switch(c)
	{
	case 'h':
	  h = atof(optarg);
	  break;
	case 's':
	  if (strcmp(optarg,"0") == 0) adaptive = 0 ;
	  else
	    {
	      adaptive = 1 ;
	      sensitivity = atof(optarg) ;
	      if (sensitivity < 0 || sensitivity > 1)
		message('E',"Sensitivity must have value in range [0,1].\n");
	    }
	  break;
	case 'k':
	  if (strcmp(optarg, "gauss") == 0) krnl=gauss;
	  else if (strcmp(optarg, "triang") == 0) krnl=triang;
	  else if (strcmp(optarg, "naive") == 0) krnl=naive;
	  else message('U', "Inexisting option for -k.\n");
	  break;
	case 'F':
	  w_pdat = 1;
	  break;
	case 'E':
	  w_E = 1;
	  break;
	default:
	  message('U', "Options usage.\n");
	}
    }

  if ((argc - optind) != 4) message('U', "Wrong number of arguments.\n") ;

  strcpy(data_file, argv[optind]) ;
  ndim = atoi(argv[optind+1]);
  dr = atof(argv[optind+2]) ;
  strcpy(rname, argv[optind+3]) ;


}



/*Reads point[p].r from data_file*/
void read_data(void)
{
  int p, d, c, b, wc ;
  FILE *fp_data ;
  string line ;

  /* One pass over data just to count the number of points (lines) */
  if ((fp_data = fopen(data_file, "r"))==NULL)
    message('E', "File %s can not be found in running directory.\n", data_file) ;
  npoints = 0 ;
  while (fgets(line,MAXSTR-1,fp_data) != NULL) npoints++ ;
  fclose(fp_data) ;

  /* Allocate points */
  point = calloc(npoints, sizeof(point_t)) ;

  /* Read data again and assign the point coordinates */

  /*Checks if number of columns is bigger than or equal to ndim */
  fp_data = fopen(data_file, "r") ;
  wc = ((b = fgetc(fp_data)) != ' ' ) ;
  while ( (c = fgetc(fp_data)) != '\n')
    { if (c != ' ' && b == ' ') wc++;
    b = c; }
  if (wc < ndim)
    message('E', "Number of columns in %s (%d) is inferior to DIM (%d).\n",
	    data_file, wc, ndim);
  if (wc > ndim)
    message('W', "Number of columns in %s (%d) is superior to DIM (%d). First %d columns will be used.\n", data_file, wc, ndim, ndim);
  fclose(fp_data) ;

  fp_data = fopen(data_file, "r") ;
  for (p = 0 ; p < npoints ; p++)
    {
      point[p].r = calloc(ndim, sizeof(real));
      for (d = 0 ; d < ndim ; d++) 
	if (fscanf(fp_data, "%f", &(point[p].r[d])) == 0 )
	  message ('E',"scanf conversion\n");
      /*Overrides exceeding columns if they exist*/
      while ((c = fgetc(fp_data)) != '\n');
    }
  fclose(fp_data) ;
}


/*Determine h (Silverman)*/
/*2019-09-11: Correction*/
void h_rule_of_thumb(void)
{

  real mean, mean2, std_dev=0, aux;
  int p, d;

  for (d=0; d<ndim; d++)
    {
      /*Determine arithmetic mean*/
      mean = 0 ;
      mean2 = 0 ;
      for (p=0; p<npoints; p++) 
	{
	  mean += point[p].r[d] ;
	  mean2 += point[p].r[d]*point[p].r[d] ;
	} 
      mean /= npoints ;
      mean2 /= npoints ;
      /*Determine average marginal variance (Silverman)*/
      std_dev += mean2 - mean*mean ;  
    }

  /*Determine standard deviation from average marginal variance (Silverman)*/

  std_dev /= ndim ;
  
  std_dev = sqrt(std_dev);

  printf("std_dev =  %g\n", std_dev);

  /*Determine h optimal*/

  aux = 1./(ndim+4);

  if (ndim==1) h = 1.06 * pow(npoints, -aux) ;

  if (ndim==2) h = 0.96 * pow(npoints, -aux) ;

  if (ndim > 2) h = pow(4./(2*ndim*npoints + npoints), aux) ;

  printf("hopt = %g\n", h);

  h *= std_dev;

}



void make_grid(void)
{
  int p, d ;
  int k, *iv ;  /* change to long */
  real mid, log_nbins=0.0, size ; /*SC 2013-10-18*/

  iv = calloc(ndim, sizeof(int));
  
  /* Compute grid bounds. */
  for (d = 0 ; d < ndim ; d++)
  {
    rmin[d] = +1e10 ;
    rmax[d] = -1e10 ;
  }
  for (p = 0 ; p < npoints ; p++)
    for (d = 0 ; d < ndim ; d++)
    {
      rmin[d] = min(rmin[d], point[p].r[d]) ;
      rmax[d] = max(rmax[d], point[p].r[d]) ;
    }
  for (d = 0 ; d < ndim ; d++)
  {
    /*SC 2013-10-18: fix truncation problems*/
    size = (rmax[d]-rmin[d])/dr ;
    if ( (int) (size + 0.5) - size < 0.00001 && (int) (size + 0.5) - size > 0)
      gsize[d] = ((int) (size + 0.5)) + GRID_BORDER ;
    else
      gsize[d] = ((int) size) + GRID_BORDER ;
    /*SC 2013-10-18: old way 
      gsize[d] = (int) ((rmax[d]-rmin[d])/dr) + GRID_BORDER ;*/
    mid = 0.5 * (rmin[d] + rmax[d]) ;
    rmin[d] = mid - 0.5 * gsize[d] * dr ;
    rmax[d] = mid + 0.5 * gsize[d] * dr ;
  }
  for (d = 0 ; d < ndim ; d++)
    fprintf(fp_log, "rmin[%d] = %g ,  rmax[%d] = %g , gsize[%d] = %d\n",
	    d, rmin[d], d, rmax[d], d, gsize[d]) ;

  /* Allocate bins */
  nbins = 1 ;
  for (d = 0 ; d < ndim ; d++) nbins *= gsize[d] ;
  bin = calloc(nbins, sizeof(bin_t)) ;
  fprintf(fp_log, "nbins = %d\n", nbins) ;

  /*Important checking of number of bins fitting in int size*/
  for (d = 0 ; d < ndim ; d++) log_nbins += log(gsize[d]) ;
  if (log_nbins > log(INT_MAX))
    message('E',"Number of bins (2^%f) higher than INT_MAX size in this machine (2^%g).\n",
	    log_nbins/log(2), log(INT_MAX)/log(2)) ;

  /* Initialize bins */
  for (k = 0 ; k < nbins ; k++)
  {
    single2multi(k, &iv) ;
    bin[k].r = calloc(ndim, sizeof(real));
    for (d = 0 ; d < ndim ; d++) bin[k].r[d] = rmin[d] + (iv[d]+0.5)*dr ;
    bin[k].nelem = 0 ;
  }

  /* Assign points to bins */
  for (p = 0 ; p < npoints ; p++)
  {
    /* Assign .bin to the nearest bin */
    for (d = 0 ; d < ndim ; d++) iv[d] =  (int)((point[p].r[d]-rmin[d])/dr) ;
    k = multi2single(iv) ;
    point[p].bin = k ;
    /* Increment the number of elements of .bin */
    bin[k].nelem++ ;
  }

  /* Allocate the bin[].elem arrays */
  for (k = 0 ; k < nbins ; k++)
  {
    bin[k].elem = calloc(bin[k].nelem, sizeof(int)) ;
    /* Set the nelem again to zero, because they are needed below */
    bin[k].nelem = 0 ;
  }
  /* Assign the points in each bin (incrementing nelem again) */
  for (p = 0 ; p < npoints ; p++)
    bin[point[p].bin].elem[bin[point[p].bin].nelem++] = p ;
}


/* These functions set the correspondence between k, the bin number, and
   iv, the vector of bin coordinates in the grid. */

void single2multi(int k, int **iv)
{
  int m=1, d ;

  for (d = 0 ; d < ndim ; d++)
    (*iv)[d] = (int)(k/(nbins/(m*=gsize[d]))) % gsize[d] ;
}


int multi2single(int *iv)
{
  int d, k=0, m=1 ;

  for (d = 0 ; d < ndim ; d++) k += nbins / (m*=gsize[d]) * iv[d] ;
  return k ;

}


/* Kernel calculations */

/*real kernel_range(void)
{
  int d ;
  real kr = 0.0 ;
  real *rv ;

  rv = calloc(ndim, sizeof(real));

  do
  {
    kr += 0.1 ;
    for (d = 0 ; d < ndim ; d++) rv[d] = kr ;
  }
  while(kernel(rv) > KERNEL_CUTOFF) ;
  return kr ;
}*/

real get_kernel_range(int dim)
{
  real x=0.0, i, aux ;
  int d;

  do  
    {
      x += 0.01;
      aux = x/sqrt(2) ;
      i=1;
      for (d=1 ; d<=dim ; d++) i *= erf(aux);
/*erf(x/sqrt(2))=integral from -x to x of the normalized gaussian function*/
    }
  while (i <= 0.95);
  return x;
}

/*real kernel(real *rv)
{
  switch(krnl)
    {
    case gauss:
      return kernel_gauss(rv);
    case triang:
      return kernel_triang(rv);
    case naive:
      return kernel_naive(rv);
    default:
      message('E', "Kernel definition.\n");
    }
}*/


real kernel_gauss(real *rv)
{
  /* Constant (2*pi)^(-dim/2) */
  const real gauss_norm = pow(6.28318530717958647688, -0.5*ndim) ;
  real kd ;
  int d ;

  /* Gaussian kernel */
  kd = gauss_norm ;
  for (d = 0 ; d < ndim ; d++) kd *= invexp(0.5*(rv[d]*rv[d])) ;
  return kd ;
}

real kernel_naive(real *rv)
{
  real kd;
  int d;

  /* Naive kernel*/
  kd = 1 ;
  for (d = 0 ; d < ndim ; d++)  kd *= (rv[d]>-1 && rv[d]<1) ? 0.5 : 0;
  return kd ; 
}


real kernel_triang(real *rv)
{
  real kd;
  int d;

  /* Triangular kernel*/
  kd = 1 ;
  for (d = 0 ; d < ndim ; d++)
    kd *= (rv[d]>-1 && rv[d]<0) ? (1 + rv[d]) : (rv[d]>=0 && rv[d]<1) ? (1 - rv[d]) : 0;
  return kd ; 
}



/* This function is a piecewise rational polynomial approximation of
   exp(-x) in the intervals [0,6], [6,13] and [13,70]. Thus, it may
   give significant (even drastic) errors outside the range [0,70]. */
real invexp(real x)
{
  real x2, x3, x4 ;

  if (x > 13)
  {
    return 8.194236147130614e-10 - 1.3290994520804703e-11 * x ;
  }
  else
  {
    x2 = x * x ;
    x3 = x2 * x ;
    if (x > 6)
    {
      return (-0.0013245823657199278 + 0.00027464252539452071 * x -
              0.000019314947607346905 * x2 + 4.598224667374957e-7 * x3) /
             (1.0 - 0.5165170691890946 * x + 0.09211442135429947 * x2 -
              0.006143102546214945 * x3) ;
    }
    else
    {
      x4 = x2 * x2 ;
      return (0.9999965470613797 - 0.3960827416191208 * x +
              0.06303500815508939 * x2 - 0.00476617578304489 * x3 +
              0.00014392025197088043 * x4)/
             (1.0 + 0.6038220689877429 * x + 0.16732494517488303 * x2 +
              0.026354026827091058 * x3 + 0.00289071552898347 * x4) ;
    }
  }
}



void compute_density(real kr)
{
  real  inv_norm, inv_h, integral=0.0 ;
  int d, p, p2, m, n, k, nneig, *neig;
  int *low, *high, *iv ;
  real *rv, *pdensity ;

  low = calloc(ndim, sizeof(int));
  high = calloc(ndim, sizeof(int));
  iv = calloc(ndim, sizeof(int));
  rv = calloc(ndim, sizeof(real));

  /* This function is the time-limiting step of the whole program.
     Therefore, auxiliary variables are used containing the bin and
     point coordinates already divided by the bandwidths. This results
     in a significant speed increase. */

  inv_norm = npoints ;
  for (d = 0 ; d < ndim ; d++) inv_norm *= h ;
  inv_norm = 1.0 / inv_norm ;
  inv_h = 1.0 / h ;

  /* Compute density */
  for (p = 0 ; p < npoints ; p++)
  {
    pdensity = &(point[p].density) ;
    /* Get all neighbors */
    nneig = 1 ;
    for (d = 0 ; d < ndim ; d++)
    {
      low[d]  = max(0, (int)((point[p].r[d]-kr*h - rmin[d]) / dr)) ;
      high[d] = min(gsize[d]-1,
                    (int)((point[p].r[d]+kr*h - rmin[d]) / dr)) ;

      nneig *= high[d] - low[d] + 1 ;
    }
    neig = calloc(nneig, sizeof(int)) ;
    get_neighbors(0, low, high, &neig, iv) ;

    /* Run over all neighbors */
    for (n = 0 ; n < nneig ; n++)
    {
      k = neig[n] ;
      for (d = 0 ; d < ndim ; d++)  /* Compute difference vector */
	rv[d] = (bin[k].r[d] - point[p].r[d]) * inv_h ;
      /* Compute density at bin centers affected by point p */
      bin[k].density +=	kernel(rv) ;
      /* Compute density at point p caused by its neighbor points (p2) */
      for (m = 0 ; m < bin[k].nelem ; m++)
      {
	p2 = bin[k].elem[m] ;
	for (d = 0 ; d < ndim ; d++)  /* Compute difference vector */
	  rv[d] = (point[p].r[d] - point[p2].r[d]) * inv_h ;
	*pdensity += kernel(rv) ;
      }
    }
    free(neig) ;
  }

  /* Density at bins: product with constant factors and integral calculation */
  for (k = 0 ; k < nbins ; k++)
    integral += (bin[k].density *= inv_norm) ;
  for (d = 0 ; d < ndim ; d++) integral *= dr ;
  fprintf(fp_log, "Density integral = %g  (simple calculation)\n", integral) ;

  /* Density at points: product with constant factors */
  for (p = 0 ; p < npoints ; p++) point[p].density *= inv_norm ;
}



/* Gives bin index, neig[n], corresponding to all neighbours of a bin
   centered in the range low-high. */
void get_neighbors(int d, int *low, int *high, int **neig, int aux[])
{
  static int n ;
  int i ;

  if (d == 0) n = 0 ;
  for (i = low[d] ; i <= high[d] ; i++)
  {
    aux[d] = i ;
    if (d < ndim-1) get_neighbors(d+1, low, high, neig, aux) ;
    else (*neig)[n++] = multi2single(aux) ;
  }
}




void compute_density_adaptive(real kr)
{
  real inv_norm, integral=0.0, pden, *inv_h, *rv ;
  int d, p, p2, m, n, k, nneig, *neig, *low, *high, *iv ;
  FILE *fp_lbw ;
  string saux ;
  double gmean=0.0 ;

  inv_h = calloc(npoints, sizeof(real));
  rv = calloc(ndim, sizeof(real));
  low = calloc(ndim, sizeof(int));
  high = calloc(ndim, sizeof(int));
  iv = calloc(ndim, sizeof(int));

  /* This function is the time-limiting step of the whole program.
     Therefore, auxiliary variables are used containing the bin and
     point coordinates already divided by the bandwidths. This results
     in a significant speed increase. */

  inv_norm = 1.0 / npoints ;

  /* Compute local bandwidth factors (Silverman, 1986, section 5.3.1) */
  for (p = 0 ; p < npoints ; p++) gmean += log(point[p].density) ;
  gmean = exp(gmean/npoints) ;
  /* Allocate and assign auxiliary coordinates for the bandwidth */
  for (p = 0 ; p < npoints ; p++)
    inv_h[p] = 1.0 / (h * pow(point[p].density/gmean, -sensitivity)) ;

  for (k = 0 ; k < nbins ; k++) bin[k].density = 0.0 ;
  for (p = 0 ; p < npoints ; p++) point[p].density = 0.0 ;

  /* Compute density */
  for (p = 0 ; p < npoints ; p++)
  {
    /* Get all neighbors */
    nneig = 1 ;
    for (d = 0 ; d < ndim ; d++)
    {
      low[d]  = max(0, (int)((point[p].r[d]-kr/inv_h[p]-rmin[d]) / dr)) ;
      high[d] = min(gsize[d]-1,
		    (int)((point[p].r[d]+kr/inv_h[p]-rmin[d]) / dr)) ;
      nneig *= high[d] - low[d] + 1 ;
    }
    neig = calloc(nneig, sizeof(int)) ;
    get_neighbors(0, low, high, &neig, iv) ;

    /* Run over all neighbors */
    for (n = 0 ; n < nneig ; n++)
    {
      k = neig[n] ;
      for (d = 0 ; d < ndim ; d++)  /* Compute difference vector */
	rv[d] = (bin[k].r[d] - point[p].r[d]) * inv_h[p] ;
      /* Compute density at bin centers affected by point p */
      pden = kernel(rv) ;
      for (d = 0 ; d < ndim ; d++) pden *= inv_h[p] ;
      bin[k].density += pden ;
      /* Compute density at neighbor points p2 caused by point p */
      for (m = 0 ; m < bin[k].nelem ; m++)
      {
	p2 = bin[k].elem[m] ;
	for (d = 0 ; d < ndim ; d++)  /* Compute difference vector */
	  rv[d] = (point[p2].r[d] - point[p].r[d]) * inv_h[p] ;
	pden = kernel(rv) ;
	for (d = 0 ; d < ndim ; d++) pden *= inv_h[p] ;
	point[p2].density += pden ;
      }
    }
    free(neig) ;
  }

  /* Density at bins: product with constant factors and integral calculation */
  for (k = 0 ; k < nbins ; k++)
    integral += (bin[k].density *= inv_norm) ;
  for (d = 0 ; d < ndim ; d++) integral *= dr ;
  fprintf(fp_log, "Density integral = %g  (adaptive calculation)\n", integral) ;

  /* Density at points: product with constant factors */
  for (p = 0 ; p < npoints ; p++) point[p].density *= inv_norm ;

  /* Write local bandwidths (with density) */
  strcpy(saux, rname) ;
  fp_lbw = fopen(strcat(saux,".lbw"), "w") ;
  fprintf(fp_lbw, "#  h[i]   density[i]\n") ;
  for (p = 0 ; p < npoints ; p++)
    fprintf(fp_lbw, "%g   %g\n", 1.0/inv_h[p], point[p].density) ;
}


void compute_density_max(void)
{
  int p ;

  /* Find density maximum */
  density_max = 0.0 ;
  for (p = 0 ; p < npoints ; p++)
    if (point[p].density > density_max) density_max = point[p].density ;

}


/*Writes  rnameP.fld if ftype==P (density map) and rnameE.fld if ftype==E (energy map).*/
void write_fld(type ftype)
{
  int dim, *iv;
  FILE *fp_fld ;
  string saux ;


  strcpy(saux, rname) ;

  if (ftype == P) 
    fp_fld = fopen(strcat(saux,"P.fld"), "w") ;
  else 
    fp_fld = fopen(strcat(saux,"E.fld"), "w") ;


  /* Write header */
  fprintf(fp_fld,"%s", "# AVS field file\n") ;
  if (ftype == P)
    fprintf(fp_fld, "# Kernel-estimated density. Bandwidth = %f\n", h) ;
  else
    fprintf(fp_fld, "# Kernel-estimated energy. Bandwidth = %f\n", h) ;
  fprintf(fp_fld, "ndim=%d\n", ndim) ;
  for (dim=1 ; dim <= ndim ; dim++)
    fprintf(fp_fld, "dim%d=  %4d\n", dim, gsize[dim - 1]) ;
  fprintf(fp_fld, "nspace=%d\n", ndim) ;
  fprintf(fp_fld, "veclen=%d\n", 1) ;
  fprintf(fp_fld,"%s", "data=float\n""min_ext=") ;
  for (dim=0 ; dim < ndim ; dim++)
    fprintf(fp_fld,"%10.4f", rmin[dim]);
  fprintf(fp_fld,"%s","\n""max_ext=");
  for (dim=0 ; dim < ndim ; dim++)
    fprintf(fp_fld,"%10.4f", rmax[dim]);
  fprintf(fp_fld,"%s", "\n""field=uniform\n");
  

  /* Write the binary section separator */
  fprintf(fp_fld, "%c%c", 12, 12) ;

  /* Write binary data */
  iv = calloc(ndim, sizeof(int));
  write_binary(fp_fld , (ndim - 1), ftype, iv);

  fclose(fp_fld) ;
}


/* Recursive function that allows writting densities(ftype=P) or energies
   (ftype=E) to a binary file, corresponding to a point with dim
   dimensions. */
  void write_binary(FILE *f_fld , int dim, type ftype, int iv[])
{
  real faux;
  
  for (iv[dim] = 0 ; iv[dim] < gsize[dim] ; iv[dim]++)
    {
      if (dim == 0)
    	{
	  faux = bin[ multi2single(iv) ].density ;
	  if (ftype == E)
	    {
	      if (faux > 0)
		faux = - log(faux / density_max) ;
	      else
		faux = MAX_ENERGY ;
	    }
	  fwrite(&faux, sizeof(float), 1, f_fld) ;
	}
      else
	write_binary(f_fld , (dim-1), ftype, iv);
    }
}

void write_2D_gnuplot(type ftype)
{
  int k, *iv ;
  real raux ;
  string saux ;
  FILE *fp_gp ;

  iv = calloc(ndim, sizeof(int));

  if (ftype != P && ftype != E)
    message('E', "Wrong field type in write_2D_gnuplot().\n") ;

  strcpy(saux, rname) ;
  if (ftype == P) fp_gp = fopen(strcat(saux,"P.gp"), "w") ;
  else fp_gp = fopen(strcat(saux,"E.gp"), "w") ;

  for (iv[0] = 0 ; iv[0] < gsize[0] ; iv[0]++)
  {
    if (ndim == 2)
      {
	for (iv[1] = 0 ; iv[1] < gsize[1] ; iv[1]++)
	  {
	    k = multi2single(iv) ;
	    raux = bin[k].density ;
	    if (ftype == E)
	      {
		if (raux > 0)
		  raux = - log(raux / density_max);
		else
		  raux = MAX_ENERGY;
	      }
	    fprintf(fp_gp, "%f %f %12g\n", bin[k].r[0], bin[k].r[1], raux) ;
	  }
	fprintf(fp_gp, "\n") ;
      }
    else
      {
	k = multi2single(iv) ;
	raux = bin[k].density ;
	if (ftype == E)
	  {
	    if (raux > 0)
	      raux = - log(raux / density_max);
	    else
	      raux = MAX_ENERGY;
	  }
	fprintf(fp_gp, "%f %12g\n", bin[k].r[0], raux) ;
      }
  }
  fclose(fp_gp) ;
}

/* Writes file rname.dat with the following information: data_file,
   npoints, density_max, point[p].bin and point[p].energy. */
void write_point_data(void)
{
  FILE *fp_point ;
  string aux ;
  int p ;
  real energy;
 
  strcpy(aux, rname) ;
  fp_point = fopen(strcat(aux,".dat"), "w") ;

  fprintf(fp_point,"#DATA FILE: %s\n", data_file);
  fprintf(fp_point,"#NUM POINTS: %d\n", npoints);
  fprintf(fp_point,"#MAX DENSITY: %g\n#\n", density_max);
  fprintf(fp_point,"#%6s %49s  ENERGY(RT)\n", "FRAME", "BIN");

  for (p=0 ; p < npoints ; p++)
    {
      energy = - log (point[p].density/density_max); 
      fprintf(fp_point,"%6d %50d  %g\n", p+1, point[p].bin, energy);
    }

  fclose(fp_point);

}



int ndigits(int i)
{
  return 1 + (int) log10(i) ;
}

void message(char mtype, char *format, ...)
{
  va_list args ;
  const char cmd[] = "getdensity" ;
  const char usage[] =
    "Usage: %s [options] datafile ndim meshsize runname\n" \
    "Options:\n" \
    "  -h bandwidth : Default = Silverman's rule of thumb.\n" \
    "  -s sensitivity : Default = 0.\n" \
    "  -k kernel : gauss, triang or naive. Default = triang.\n" \
    "  -F : Writes runname.dat file with point data; getbasins requires\n" \
    "       this file.\n" \
    "  -E : Writes runnameE.fld file (with energies); getbasins does *not*\n" \
    "       require this file.\n" ;

  va_start(args, format) ;
  if (mtype != 'W' && mtype != 'E' && mtype != 'U')
    message('E', "Wrong use of message() function.\n") ;
  if (mtype == 'W') fprintf(stderr, "%s: WARNING: ", cmd) ;
  else fprintf(stderr, "%s: ERROR: ", cmd) ;
  vfprintf(stderr, format, args) ;
  va_end(args) ;
  if (mtype == 'U') fprintf(stderr, usage, cmd) ;
  if (mtype != 'W') exit(1) ;
}

