/*
This file is part of LandscapeTools, version 2.3.

Copyright (c) 2006-2019, Instituto de Tecnologia Quimica e Biologica,
Universidade Nova de Lisboa, Portugal.

LandscapeTools is free software: you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation, either version 2 of the License, or (at your
option) any later version.

LandscapeTools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with LandscapeTools.  If not, see <http://www.gnu.org/licenses/>.

For further details and info check the README file.

You can get LandscapeTools at www.itqb.unl.pt/simulation
*/


/**************************************************************************

GETBASINS: This program is meant to be used with GETDENSITY output
  files. It assigns points in a landscape to basins, calculates the energy
  minima and the free energy of the basins, analyses the inter-basin
  transitions, and produces some visualization scripts for Gnuplot and
  PyMOL.

**************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <stdarg.h>
#include <math.h>
#include <limits.h>

#define PACKAGE "LandscapeTools 2.3"
#define MAXSTR 500
#define MAX_ENERGY 1e6
#define SPHERE_SCALE 30.0
#define STICK_SCALE 1000.0
#define min(x,y) ((x)<(y) ? (x) : (y))
#define max(x,y) ((x)>(y) ? (x) : (y))

typedef float real ;
typedef char string[MAXSTR] ;
typedef struct {
  int basin ;
  real  energy ;
} bin_t ;
typedef struct {
  int bin ;
  real energy, *r ;
} point_t ;
typedef struct {
  int nelem, *elem, pmin, order ;
  real mean_energy, free_energy, *center ;
  string label ;
} basin_t ;

typedef enum {E , P} type;

void parse_arguments(int argc, char **argv) ;
void read_point_data(void) ;
void read_bin_data(type ftype) ;
void read_binary(FILE *f_fld , int dim, type ftype, int iv[]);
void read_data(void) ;
void single2multi(int k, int iv[]) ;
int multi2single(int iv[]) ;
void get_neighbors(int d, int *low, int *high, int **neig, int aux[]) ;
void compute_basins(void) ;
int assign_bin_to_basin(int k) ;
void thermo_calc(void);
void label_basins(void) ;
int compare_basins(const void *b1, const void *b2) ;
void write_basins(void) ;
void write_basins_pymol(void) ;
void write_scheme_pymol(void) ;
int ndigits(int i) ;
void message(char mtype, char *format, ...) ;

real cutoff, density_max ;
int *gsize ;
int ndim, npoints, nbins, nbasins ;
string rname, data_file ;
bin_t *bin ;
point_t *point ;
basin_t *basin ;
FILE *fp_log ;

int main(int argc, char **argv)
{
  parse_arguments(argc, argv) ;
  read_point_data() ;
  read_bin_data(P) ; 

  fp_log = stdout ;
  fprintf(fp_log, "*** GETBASINS (part of %s) ***\n", PACKAGE) ;
  fprintf(fp_log, "DIM = %d\n", ndim) ;
  fprintf(fp_log, "DATA FILE = %s\n", data_file) ;
  fprintf(fp_log, "NPOINTS = %d\n", npoints) ;
  fprintf(fp_log, "NBINS = %d\n", nbins) ;
  fprintf(fp_log, "CUTOFF = %f\n", cutoff) ;

  read_data() ;

  compute_basins() ;

  thermo_calc();

  fprintf(fp_log, "NBASINS = %d\n", nbasins) ;

  label_basins() ;

  write_basins() ;

  write_basins_pymol() ;

  write_scheme_pymol() ;

  fprintf(fp_log,
	  "Check %s-min.pdb for the point with 0 energy. If you can't find it, the mesh\nsize used is too big, meaning that the energy minimum is in a (excluded) bin\nof very high energy.\n", rname) ;
  fprintf(fp_log, "Program normal termination.\n") ;
  return 0 ;
}


/* Reads arguments */
void parse_arguments(int argc, char **argv)
{
  if (argc < 2 || argc > 3) message('U', "Wrong number of arguments.\n") ;

  strcpy(rname, argv[1]) ;

  if (argc == 3) cutoff = atof(argv[2]) ;
  else cutoff = MAX_ENERGY ;

}


/* Reads file rname.dat and assigns data_file, npoints, density_max,
   point[p].bin and point[p].energy */
void read_point_data(void)
{
  FILE *fp_dat;
  string aux;
  int p;

  strcpy(aux, rname);

  if ((fp_dat = fopen(strcat(aux,".dat"), "r")) == NULL)
    message('E', "File %s can not be found in running directory.\n", aux);

  fscanf(fp_dat, "#DATA FILE: %s\n", data_file) ;
  fscanf(fp_dat, "#NUM POINTS: %d\n", &npoints) ;
  fscanf(fp_dat, "#MAX DENSITY: %g\n#\n", &density_max) ;
  fscanf(fp_dat, "#%*s%*s%*s\n") ;

  if (npoints > 99999)
    message('W', "Number of points > 99999. See Note in README file.\n");

  point=calloc(npoints,sizeof(point_t));
  
  for (p=0 ; p < npoints ; p++)
      fscanf(fp_dat,"%*d%d%g\n", &(point[p].bin), &(point[p].energy));

  fclose(fp_dat);

}

/* Reads file rnameX.fld and assigns ndim, gsize[d], nbins and
   bin[k].energy. ftype=P if rnameP.fld and ftype=E if rnameE.fld. */
void read_bin_data(type ftype)
{
  int dim, *iv;
  FILE *fp_fld ;
  string aux;
  real log_nbins=0.0 ;

  strcpy(aux, rname);

  if (ftype == P)
    {
      if ((fp_fld = fopen(strcat(aux,"P.fld"), "rb")) == NULL)
	message('E', "File %s can not be found in running directory.\n", aux);
    }
  else
      if ((fp_fld = fopen(strcat(aux,"E.fld"), "rb")) == NULL)
	message('E', "File %s can not be found in running directory.\n", aux);


  /* Read data in header */
  fscanf(fp_fld, "# AVS field file\n") ;
  fscanf(fp_fld, "# Kernel-estimated %*s Bandwidth = %*f\n") ;
  fscanf(fp_fld, "ndim=%i\n", &ndim) ;
  gsize=calloc(ndim, sizeof(int));
  for (dim=1 ; dim <= ndim ; dim++)
    fscanf(fp_fld, "dim%*d=  %d\n",  &(gsize[dim - 1])) ;
  fscanf(fp_fld, "nspace=%*d\n") ;
  fscanf(fp_fld, "veclen=%*d\n") ;
  fscanf(fp_fld, "data=float\n") ;
  fscanf(fp_fld, "min_ext=");
  for (dim=0 ; dim < ndim ; dim++)
    fscanf(fp_fld,"%*f");
  fscanf(fp_fld, "\n");
  fscanf(fp_fld, "max_ext=");
  for (dim=0 ; dim < ndim ; dim++)
    fscanf(fp_fld,"%*f");
  fscanf(fp_fld, "\n");
  fscanf(fp_fld, "field=uniform\n") ;


  /* Important checking of number of bins fitting in long int size */
  for (dim = 0 ; dim < ndim ; dim++) log_nbins += log(gsize[dim]) ;
  if (log_nbins > log(INT_MAX))
    message('E',"Number of bins (2^%f) higher than INT_MAX size in this machine (2^%g).\n",
	    log_nbins/log(2), log(INT_MAX)/log(2)) ;

  nbins=1;
  for (dim = 0 ; dim < ndim ; dim++) nbins *= gsize[dim] ;
  bin=calloc(nbins, sizeof(bin_t));

 /* Read binary data separator*/
  fscanf(fp_fld,"\f\f");

  /* Read binary data */
  iv=calloc(ndim, sizeof(int));
  read_binary(fp_fld , (ndim-1), ftype, iv);
  free(iv);

  fclose(fp_fld) ;

}


/* Recursive function that allows reading densities(ftype=P) or energies
   (ftype=E) from a binary file, corresponding to a point with dim
   dimensions. */
void read_binary(FILE *f_fld , int dim, type ftype, int iv[])
{

  float faux;
  for (iv[dim] = 0 ; iv[dim] < gsize[dim] ; iv[dim]++)
    {
      if (dim == 0)
        {
          fread(&faux, sizeof(float), 1, f_fld) ;
	  if (ftype == E)
	    bin[ multi2single(iv) ].energy = faux;
	  else
	   bin[ multi2single(iv) ].energy = -log (faux/density_max) ;
        }
      else
        read_binary(f_fld , (dim-1) , ftype,  iv);
    }
}


/* Reads point[p].r from data_file */
void read_data(void)
{
  int p, d, c, b, wc ;
  FILE *fp_data ;

 
  if ((fp_data = fopen(data_file, "r")) == NULL)
    message('E', "File %s can not be found in running directory.\n", data_file);


  /* Checks if number of columns is bigger than or equal to ndim */
  wc = ((b = fgetc(fp_data)) != ' ' ) ;
  while ( (c = fgetc(fp_data)) != '\n')
    {
      if (c != ' ' && b == ' ') wc++;
      b = c;
    }
  if (wc < ndim)
    message('E', "Number of columns in %s (%d) is inferior to ndim (%d).\n",
	    data_file, wc, ndim);
  if (wc > ndim)
    message('W', "Number of columns in %s (%d) is superior to ndim (%d). First %d columns will be used.\n", data_file, wc, ndim, ndim);
  fclose(fp_data) ;

  /* Reads data */
  fp_data = fopen(data_file, "r") ;
  for (p = 0 ; p < npoints ; p++)
    {
      point[p].r = calloc(ndim, sizeof(real));
      for (d = 0 ; d < ndim ; d++)
        if (fscanf(fp_data, "%f", &(point[p].r[d])) == 0 )
	  message('E',"scanf conversion");
      /* Overrides exceeding columns if they exist */
      while ((c = fgetc(fp_data)) != '\n');
    }
  fclose(fp_data) ;
}


/* These functions set the correspondence between k, the bin number, and
   iv, the vector of bin coordinates in the grid. */

void single2multi(int k, int iv[])
{
  int m=1, d ;

  for (d = 0 ; d < ndim ; d++)
    iv[d] = (int)(k/(nbins/(m*=gsize[d]))) % gsize[d] ;
}

int multi2single(int iv[])
{
  int d, k=0, m=1 ;

  for (d = 0 ; d < ndim ; d++) k += nbins / (m*=gsize[d]) * iv[d] ;
  return k ;
}


/* Assignment of bins and points to basins. Assignment of basin elements. */
void compute_basins(void)
{
  int  k, p, b, e, pmin ;
  real emin ;


  for (k = 0 ; k < nbins ; k++) bin[k].basin = -1 ;
  nbasins = 0 ;
  for (k = 0 ; k < nbins ; k++)
    bin[k].basin = assign_bin_to_basin(k) ;



  /* Allocate basin and initialize basin[].nelem */
  basin = calloc(nbasins, sizeof(basin_t)) ;
  for (b = 0 ; b < nbasins ; b++)
  {
    basin[b].nelem = 0 ;
    /* the next 2 assignments will be changed later by label_basins() */
    basin[b].order = b ;
    sprintf(basin[b].label, "%0*d", ndigits(nbasins), b) ;
  }


  /* Count points per basins */
  for (p = 0 ; p < npoints ; p++) 
    {
      b = bin[point[p].bin].basin ;
      if (b != -2) basin[b].nelem++ ;
    }


  /* Allocate the basin[].elem arrays */
  for (b = 0 ; b < nbasins ; b++)
  {
    if (basin[b].nelem > 0)
      basin[b].elem = calloc(basin[b].nelem, sizeof(int)) ;
    /* Set the nelem again to zero, because they'll be needed below */
    basin[b].nelem = 0 ;
  }



  /* Assign points to basins (incrementing nelem again) */
  for (p = 0 ; p < npoints ; p++)
  {
    b = bin[point[p].bin].basin ;
    if (b != -2) basin[b].elem[basin[b].nelem++] = p ;
  }


  /* Compute energy minima */
  for (b = 0 ; b < nbasins ; b++)
    if (basin[b].nelem > 0)
      { 
	emin = MAX_ENERGY ;
	pmin = -1 ;
	for (e = 0 ; e < basin[b].nelem ; e++)
	  {
	    p = basin[b].elem[e] ;
	    if (point[p].energy < emin)
	      {
		pmin = p ;
		emin = point[p].energy ;
	      }
	  }
	basin[b].pmin = pmin ;
      }
}


/* Assigns bin to basin by searching minimum among neighbours and following
   a path till a local minimum is found. All bins in the path are
   attributed to that local minimum. */
int assign_bin_to_basin(int k)
{
  int d, n, down, is_min, *kv, *low, *high, *iv ;
  static int nneig ;
  static int *neig ;
  real energy_here, energy_neigh, energy_down ;


  energy_here = bin[k].energy ;

  /* If bin[k].basin was already assigned, return it */
  if (bin[k].basin != -1) return bin[k].basin ;

  /* If energy is "infinite", return -2 */
  if (energy_here > MAX_ENERGY/2) return -2 ;

  /* For the others, scan the neighbors */
  kv = calloc(ndim, sizeof(int));
  single2multi(k, kv) ;
  is_min = 1 ;
  energy_down = MAX_ENERGY ;
  down = -1 ;

  /* Get all neighbors */
  iv = calloc(ndim, sizeof(int));
  low = calloc(ndim, sizeof(int));
  high = calloc(ndim, sizeof(int));
  nneig = 1 ;
  for (d = 0 ; d < ndim ; d++)
  {
    low[d]  = max(0, kv[d]-1) ;
    high[d] = min(gsize[d]-1, kv[d]+1) ;
    nneig *= high[d] - low[d] + 1 ;
  }

  neig = calloc(nneig, sizeof(int)) ;
  get_neighbors(0, low, high, &neig, iv) ;
  free(kv);
  free(iv);
  free(low);
  free(high);


  /* Run over all neighbors */
  for (n = 0 ; n < nneig ; n++)
    if (neig[n] != k)
      if ((energy_neigh = bin[neig[n]].energy) < energy_here)
      {
	is_min = 0 ;
	if (energy_neigh < energy_down)
	{
	  energy_down = energy_neigh ;
	  down = neig[n] ;
	}
      }
  free(neig) ;


  if (is_min) return (bin[k].basin = nbasins++) ;
  else return bin[k].basin = assign_bin_to_basin(down) ;
}



/* Gives bin index, neig[n], corresponding to all neighbours of a bin
   centered in the range low-high. */
void get_neighbors(int d, int *low, int *high, int **neig, int aux[])
{
  static int n ;
  int i ;

  if (d == 0) n = 0 ;
  for (i = low[d] ; i <= high[d] ; i++)
  {
    aux[d] = i ;
    if (d < ndim-1) get_neighbors(d+1, low, high, neig, aux) ;
    else (*neig)[n++] = multi2single(aux) ;
  }
}


/* Calculates thermodynamic properties with or without an energy cutoff. */
void thermo_calc(void)
{
  int *cutoff_nelem , cutoff_npoints = 0 ;
  int b, d, e, p ;
 

  if ( cutoff < MAX_ENERGY )
    {

      /* Compute point-based free energy, mean energy, and geometric center
	 with cutoff */
      cutoff_nelem=calloc(nbasins, sizeof(int));
      
      for (b = 0 ; b < nbasins ; b++)
	if (basin[b].nelem > 0)
	  {
	    cutoff_nelem[b]=0;
	    basin[b].mean_energy = 0.0 ;
	    basin[b].center = calloc(ndim, sizeof(real));
	    for (d = 0 ; d < ndim ; d++) basin[b].center[d] = 0.0 ;
	    if ( point[basin[b].pmin].energy <= cutoff )
	      {
		for (e = 0 ; e < basin[b].nelem ; e++)
		  {
		    p = basin[b].elem[e] ;
		    if (point[p].energy > cutoff) continue ;
		    basin[b].mean_energy += point[p].energy ;
		    for (d = 0 ; d < ndim ; d++)
		      basin[b].center[d] += point[p].r[d] ;
		    cutoff_nelem[b]++ ;
		  }
		basin[b].mean_energy /= cutoff_nelem[b] ;
		for (d = 0 ; d < ndim ; d++)
		  basin[b].center[d] /= cutoff_nelem[b] ;
	      }
	    cutoff_npoints += cutoff_nelem[b] ;
	  }
      for (b = 0 ; b < nbasins ; b++)
	if (basin[b].nelem > 0)
	  { 
	    if ( point[basin[b].pmin].energy <= cutoff )
	      basin[b].free_energy =
		-log (cutoff_nelem[b]/(cutoff_npoints*density_max));
	    else
	      basin[b].free_energy =
		-log(basin[b].nelem / (npoints * density_max));
	  }
    }

  else

    /* Compute point-based free energy, mean energy, and geometric center */
    for (b = 0 ; b < nbasins ; b++)
      if (basin[b].nelem > 0)
	{
	  basin[b].free_energy =
	    -log(basin[b].nelem / (npoints * density_max)) ;
	  basin[b].mean_energy = 0.0 ;
	  basin[b].center = calloc(ndim, sizeof(real));
	  for (d = 0 ; d < ndim ; d++) basin[b].center[d] = 0.0 ;
	  for (e = 0 ; e < basin[b].nelem ; e++)
	    {
	      p = basin[b].elem[e] ;
	      basin[b].mean_energy += point[p].energy ;
	      for (d = 0 ; d < ndim ; d++) basin[b].center[d] += point[p].r[d] ;
	    }
	  basin[b].mean_energy /= basin[b].nelem ;
	  for (d = 0 ; d < ndim ; d++) basin[b].center[d] /= basin[b].nelem ;
	}
}



/* Tag basins with sorted labels (see compare_basins) */
void label_basins(void)
{
  int b, *index ;

  index = calloc(nbasins, sizeof(int)) ;
  for (b = 0 ; b < nbasins ; b++) index[b] = b ;
  
  /* Try to solve in some other way... */
  for (b = 0 ; b < nbasins ; b++)
    if (basin[b].nelem == 0) basin[b].free_energy = MAX_ENERGY ;

  qsort(index, nbasins, sizeof(int), compare_basins) ;
  for (b = 0 ; b < nbasins ; b++)
  {
    basin[index[b]].order = b ;
    sprintf(basin[index[b]].label, "%0*d", ndigits(nbasins), b) ;
  }
  free(index) ;
}

/* Different properties can be used to sort the basins. */
int compare_basins(const void *vi, const void *vj)
{
  const int *i = vi ;
  const int *j = vj ;
  real diff ;

  /* Use Emin to sort basins */
  /* diff = point[basin[*i].pmin].energy - point[basin[*j].pmin].energy ; */

  /* Use free energy A to sort basins */
  diff = basin[*i].free_energy - basin[*j].free_energy ;

  if (diff < 0.0) return -1 ;
  else if (diff > 0.0) return +1 ;
  else return 0 ;
}

void write_basins(void)
{
  int ord, b, e, p ;
  FILE *fp_bas, *fp_ndx, *fp_min, *fp_thermo, *fp_trj ;
  char saux[2*MAXSTR] ;

  /* Open .ndx file */
  strcpy(saux, rname) ;
  fp_ndx = fopen(strcat(saux,".ndx"), "w") ;
  /* Open .min PDB file */
  strcpy(saux, rname) ;
  fp_min = fopen(strcat(saux,"-min.pdb"), "w") ;
  /* Open .thermo file */
  strcpy(saux, rname) ;
  fp_thermo = fopen(strcat(saux,".thermo"), "w") ;

  fprintf(fp_thermo, "# basin    A/RT   <E>/RT    S/R   Emin/RT    %%\n") ;
  fprintf(fp_thermo, "#----------------------------------------------\n") ;

  /* The following double loop is not very pretty in terms of
     programming, but it ensures that things are written in the order
     implied by the basin labels (see label_basins()). */
  for (ord = 0 ; ord < nbasins ; ord++)
  for (b = 0 ; b < nbasins ; b++)
  {
    if (basin[b].order != ord) continue ;
    if (basin[b].nelem == 0)
      {
	fprintf(fp_log, "BASIN %4s with empty bins\n", basin[b].label) ;
	break;
      }
    /* Write PDB line to .min (with points starting at 1, not at 0) */
    fprintf(fp_min,
	    "ATOM  %5d  H   BAS  %4s    %8.3f%8.3f%8.3f  1.00%6.2f\n",
	    basin[b].pmin+1, basin[b].label,
	    point[basin[b].pmin].r[0],
	    (ndim<2 ? 0.0 : point[basin[b].pmin].r[1]),
	    (ndim<3 ? 0.0 : point[basin[b].pmin].r[2]),
	    point[basin[b].pmin].energy) ;

    /* Ignore basins with Emin>cutoff */
    if (point[basin[b].pmin].energy > cutoff) break ;

    /* Write header in .ndx file */
    fprintf(fp_ndx, "[ bas%s ]\n", basin[b].label) ;

    /* Open *-bas*.pdb PDB file */
    sprintf(saux, "%s-bas%s.pdb", rname, basin[b].label) ;
    fp_bas = fopen(saux, "w") ;

    for (e = 0 ; e < basin[b].nelem ; e++)
    {
      p = basin[b].elem[e] ;

      /* Ignore points with E>cutoff */
      if (point[p].energy > cutoff) continue ;

      /* Write .ndx line (with points starting at 1, not at 0) */
       fprintf(fp_ndx, "%d\n", p+1) ;

      /* Write PDB line (with points starting at 1, not at 0) */
      fprintf(fp_bas,
	      "ATOM  %5d  H   BAS  %4s    %8.3f%8.3f%8.3f  1.00%6.2f\n",
	      p+1, basin[b].label, point[p].r[0],
	      (ndim<2 ? 0.0 : point[p].r[1]),
	      (ndim<3 ? 0.0 : point[p].r[2]), point[p].energy) ;
    }

    /* Write group-separating newline in .ndx file */
    fprintf(fp_ndx, "\n") ;

    /* Write thermodynamic properties (and Emin) */
    fprintf(fp_thermo, "%6s  %7.2f %7.2f %7.2f %7.2f %7.2f\n",
	    basin[b].label, basin[b].free_energy, basin[b].mean_energy,
	    basin[b].mean_energy - basin[b].free_energy,
	    point[basin[b].pmin].energy,
	    basin[b].nelem * 100 / (float) npoints) ;

    fclose(fp_bas) ;
  }
  fclose(fp_min) ;
  fclose(fp_thermo) ;
  fclose(fp_ndx) ;

  /* Write basin trajectory */
  /* Open .traj file */
  strcpy(saux, rname) ;
  fp_trj = fopen(strcat(saux,".traj"), "w") ;
  for (p = 0 ; p < npoints ; p++)
    {
      b = bin[point[p].bin].basin ;
      if (b != -2) 
	fprintf(fp_trj, "%s\n", basin[b].label) ;
      else
	fprintf(fp_trj, "%s\n", "HIGH-ENERGY") ;
    }
  fclose(fp_trj) ;
}

void write_basins_pymol(void)
{
  int b, ord, ncolors=9 ;
  const char *color[] = {"gray", "red", "green", "blue", "yellow",
			 "magenta", "cyan", "orange", "wheat"} ;
  string saux ;
  FILE *fp_pymol ;
  int i ;
  real cnt ;

  strcpy(saux, rname) ;
  fp_pymol = fopen(strcat(saux,".pml"), "w") ;

  fprintf(fp_pymol, "# Adjust nonbonded_size and sphere_scale as needed.\n") ;
  fprintf(fp_pymol, "set nonbonded_size, 0.05\n") ;
  fprintf(fp_pymol, "set sphere_scale, 0.2\n") ;

  /* Stuff for .bas* PDB files (see write_basins() on this double loop). */
  for (ord = 0 ; ord < nbasins ; ord++)
  for (b = 0 ; b < nbasins ; b++)
  {
    if (basin[b].order != ord) continue ;
    if (basin[b].nelem == 0) break;
    if (point[basin[b].pmin].energy > cutoff) break ;

    fprintf(fp_pymol, "load %s-bas%s.pdb\n", rname, basin[b].label) ;
    if (b != nbasins)
      fprintf(fp_pymol, "color %s, %s-bas%s\n",
	      color[b % ncolors], rname, basin[b].label) ;
    else
      fprintf(fp_pymol, "color white, %s-bas%s\n", rname, basin[b].label) ;
  }

  /* Stuff for .min PDB file. */
  fprintf(fp_pymol, "load %s-min.pdb\n", rname) ;
  fprintf(fp_pymol, "show spheres, %s-min\n", rname) ;
  fprintf(fp_pymol, "load %sE.fld\n", rname) ;

  /* Choice of contour values is more-or-less arbitrary... */
  for (i=0 ; i<6 ; i++)
  {
    cnt = 1.0 + i*0.5 ;
    fprintf(fp_pymol, "isomesh %sE_%g, %sE, %g\n", rname, cnt, rname, cnt) ;
  }

  fclose(fp_pymol) ;
}

void write_scheme_pymol(void)
{
  int b, ord, b2, ord2, p, bnow, bold, pold = npoints-1, **ntrans ;
  FILE *fp_pdb, *fp_pymol ;
  string saux ;

  /* Allocate ntrans (not very efficient: most elements will remain 0) */
  ntrans = calloc(nbasins, sizeof(int *)) ;
  for (b = 0 ; b < nbasins ; b++)
    if (basin[b].nelem > 0) ntrans[b] = calloc(nbasins, sizeof(int)) ;

  /* Count number of transitions between basins */

  for (p = 0 ; p < npoints ; p++)
    {
      b = bin[point[p].bin].basin ;
      if (b != -2) 
	{
	  bold = b ;
	  pold = p;
	  break;
	}
    }
      
  for (p = pold+1 ; p < npoints ; p++)
  {
    b = bin[point[p].bin].basin ;
    if (b != -2)
      {
	bnow = b ;
	if (bnow != bold && p == pold+1) ntrans[bold][bnow]++ ;
	bold = bnow ;
	pold = p;
      }
  }

  /* Open PDB file */
  strcpy(saux, rname) ;
  fp_pdb = fopen(strcat(saux,"S.pdb"), "w") ;
  /* Open pymol file */
  strcpy(saux, rname) ;
  fp_pymol = fopen(strcat(saux,"S.pml"), "w") ;

  /* Write to *S.pml the stuff to load *S.pdb and put spheres and sticks */
  fprintf(fp_pymol, "# Adjust sphere_scale and stick_scale as needed.\n") ;
  fprintf(fp_pymol, "sphere_scale=%g\n", SPHERE_SCALE) ;
  fprintf(fp_pymol, "stick_scale=%g\n", STICK_SCALE) ;
  fprintf(fp_pymol, "load %sS.pdb\n", rname) ;
  fprintf(fp_pymol, "show spheres, %sS\n", rname) ;
  fprintf(fp_pymol, "show sticks, %sS\n", rname) ;

  /* Write individual (ie, non-pairwise) stuff */
  for (ord = 0 ; ord < nbasins ; ord++)
  for (b = 0 ; b < nbasins ; b++)
  {
    if (basin[b].order != ord) continue ;
    if (basin[b].nelem == 0) break;
    if (point[basin[b].pmin].energy > cutoff) break ;

    /* Write ATOM line to *S.pdb */
    fprintf(fp_pdb,
	    "ATOM  %5d  H   BAS  %4d    %8.3f%8.3f%8.3f  1.00%6.2f\n",
	    basin[b].order, basin[b].order,
	    basin[b].center[0], basin[b].center[1],
	    (ndim<3 ? 0.0 : basin[b].center[2]), basin[b].free_energy) ;

    /* Write to *S.pml the stuff to scale spheres according to
       basin[].nelem (ie, to free energy) */
    fprintf(fp_pymol, "create bas%s, (%sS and resi %d)\n",
	   basin[b].label, rname, basin[b].order) ;
    fprintf(fp_pymol,
	    "cmd.set('sphere_scale', str(sphere_scale*%g), 'bas%s')\n",
    	    pow(basin[b].nelem / (real)(npoints), 1.0/3.0),
	    basin[b].label) ;
  }

  /* Write pairwise stuff */
  for (ord = 1 ; ord < nbasins ; ord++)
  for (b = 0 ; b < nbasins ; b++)
  {
    if (basin[b].order != ord) continue ;
    if (basin[b].nelem == 0) break;
    if (point[basin[b].pmin].energy > cutoff) break ;
    for (ord2 = 0 ; ord2 < ord ; ord2++)
    for (b2 = 0 ; b2 < nbasins ; b2++)
    {
      if (basin[b2].order != ord2) continue ;
      if (basin[b2].nelem == 0) break;
      if (point[basin[b2].pmin].energy > cutoff) break ;
      if (ntrans[b][b2] == 0 && ntrans[b2][b] == 0) break ;

      /* Write CONECT line to *S.pdb */
      fprintf(fp_pdb, "CONECT%5d%5d\n", basin[b].order, basin[b2].order) ;

      /* Display and scale cylinders for transitions */
      fprintf(fp_pymol, "create pair%s_%s, (%sS and (resi %d or resi %d))\n",
	      basin[b].label, basin[b2].label,
	      rname, basin[b].order, basin[b2].order) ;
      fprintf(fp_pymol,
	      "cmd.set('stick_radius', str(stick_scale*%g), 'pair%s_%s')\n",
	      (ntrans[b][b2] + ntrans[b2][b]) / (real)(npoints-1),
	      basin[b].label, basin[b2].label) ;
    }
  }

  /* Write to *S.pml the stuff to hide things */
  fprintf(fp_pymol, "hide everything, %sS\n", rname) ;
  fprintf(fp_pymol, "hide nonbonded, all\n") ;
  fprintf(fp_pymol, "hide lines, all\n") ;
  fprintf(fp_pymol, "hide spheres, pair*\n") ;
  fprintf(fp_pymol, "zoom all\n") ;

  fclose(fp_pymol) ;
  fclose(fp_pdb) ;
}


int ndigits(int i)
{
  return 1 + (int) log10(i) ;
}


void message(char mtype, char *format, ...)
{
  va_list args ;
  const char cmd[] = "getbasins" ;
  const char usage[] =
    "Usage: %s runname [cutoff]\n" ;

  va_start(args, format) ;
  if (mtype != 'W' && mtype != 'E' && mtype != 'U')
    message('E', "Wrong use of message() function.\n") ;
  if (mtype == 'W') fprintf(stderr, "%s: WARNING: ", cmd) ;
  else fprintf(stderr, "%s: ERROR: ", cmd) ;
  vfprintf(stderr, format, args) ;
  va_end(args) ;
  if (mtype == 'U') fprintf(stderr, usage, cmd) ;
  if (mtype != 'W') exit(1) ;
}

