#!/usr/bin/awk -f

#USAGE: occ_to_prot.awk sites_file occ_file(s)

#This script converts the info in the .occ files (tautomeric states) in
#protonations for each titrating site, writing a file for each. It needs to
#read a .sites file to make the assignment of the protonations to the
#sites. A file with the total protonation is also written.

BEGIN{

    usage = "Usage: occ_to_prot.awk sites_file occ_file(s)"
    if (ARGC < 3) {
	print "Wrong number of arguments.\n" usage > "/dev/stderr"
	exit 1
    } 
    
    #Read sites
    sites_file=ARGV[1]
    i=0
    t=0
    while (getline < sites_file) {
	#Assign site name to index
	name[++i]=substr($2,1,index($2,"t")-1) "-" $1
    }
    close(sites_file)
    
}

#Read the occ file(s)
NR>FNR{
    
    ++t
    #Convert state in protonation and assign it to index
    for (i=1; i<=NF; i++){
	switch (name[i]) {
	case /^(ASP|GLU|CT)/:
	    ($i==4) ? a=0 : a=1; prot[t,i]=a
	    break
	case /^CYS/:
	    ($i==3) ? a=0 : a=1; prot[t,i]=a
	    break
	case /^TYR/:
	    ($i==2) ? a=0 : a=1; prot[t,i]=a
	    break
	case /^(HIS|NTPRO)/:
	    ($i==2) ? a=1 : a=0; prot[t,i]=a 
	    break
	case /^(LYS|NT)/:
	    ($i==3) ? a=1 : a=0; prot[t,i]=a
	    break
	default:
	    printf "ERROR: Unknown type - %s.\n", name[i] > "/dev/stderr" ; exit 1
	}
    }
    
}

END{
    n=t
    for (s in name)
    {
	#Write file for each site with frame and protonation
	printf "#Cycle and Protonation of %s\n", name[s]  > name[s] ".prot"
	for (t=1; t<=n; t++) 
	    printf "%d %d\n", t, prot[t,s] > name[s] ".prot"
    }

    #Write file with frame and total protonation
    printf "#Cycle and Total Protonation\n"  > "total.prot"
    for (t=1; t<=n; t++)
    {
      sum = 0 ;
      for (s in name) sum += prot[t,s] ;
      printf "%d %d\n", t, sum > "total.prot"
    }
}

