/*
This file is part of MCRP, version 1.3.

Copyright (c) 2000-2019, Instituto de Tecnologia Quimica e Biologica,
Universidade Nova de Lisboa, Portugal.

MCRP is free software: you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation, either version 2 of the License, or (at your
option) any later version.

MCRP is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with MCRP.  If not, see <http://www.gnu.org/licenses/>.

For further details and info check the README file.

You can get MCRP at www.itqb.unl.pt/simulation
*/


#define VERSION "1.3"

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <stdarg.h>

#define STRSIZE 50
#define LINESIZE 500
#define MAXDELTA 50
#define MAXNPKHALFS 5
#define LN10 2.302585093

void parse_arguments(int argc, char **argv) ;
void usage(void) ;
void make_initializations(void) ;
void read_pkint(void) ;
void read_inter(void) ;
void select_pairs(void) ;
void initialize_EpH_point(float E, float pH) ;
void initialize_E_line(float E) ;
void mc_step(float E, float pH) ;
int  accept_move(float dU) ;
void compute_statistics(int t) ;
void write_EpH_point(float E, float pH) ;
void write_E_line(float E) ;
float invexp(float x) ;
void clean_memory(void) ;
double sqrtp(double x) ;
void strsplit(char *s, const char *delim) ;
void error(char errtype, char *format, ...) ;


int nsites, npairs, pHsteps, Esteps, taumax, nset, nmicro,
    compute_set, compute_pair_corr, compute_errors, compute_energetics,
    avgP2, avgR2, avgPR, nsitesP, nsitesR, nsubstr ;
long mcsteps, eqsteps, seed ;
int  *state, *z0, *z, *pair1, *pair2, *titration, *frozen, *avg, *npkhalfs,
     **avgpair, **cf, **cc, **buf, *setsite, *avgmicro, *binP, *binR ;
char *site_type ;
float pHmin, pHmax, dpH, Emin, Emax, dE, freeze_cutoff, T, couple_min,
      min_corr, energy0 ;
float **w, *pkint, **pkhalf, *min_inter, *max_inter, *u, *pmean ;
double avgU, avgU2, *avgUn ;
char pkint_file[STRSIZE], inter_file[STRSIZE], cmd[STRSIZE], **site_name,
     sset[STRSIZE], **substr ;

float kBoltz_au  = 5.98435e-6 ;  /*  e^2/(Angstrom*K)  */
float kBoltz_meV = .0861734 ;    /*  meV/K  */


int main(int argc, char **argv)
{
  int i, j, t ;
  float pH, E ;

  parse_arguments(argc, argv) ;
  make_initializations() ;
  read_pkint() ;
  read_inter() ;
  select_pairs() ;

  for (i = 0 ; i < Esteps ; i++)
  {
    E = Emin + i * dE ;
    initialize_E_line(E) ;
    for (j = 0 ; j < pHsteps ; j++)
    {
      pH = pHmin + j * dpH ;
      initialize_EpH_point(E, pH) ;
      for (t = 0 ; t < eqsteps ; t++) mc_step(E, pH) ;
      for (t = 0 ; t < mcsteps ; t++)
      {
	mc_step(E, pH) ;
 	compute_statistics(t) ;
      }
      write_EpH_point(E, pH) ;
    }
    write_E_line(E) ;
  }

  printf("f") ;
  for (i = 0 ; i < nsites ; i++) printf(" %1d", state[i]) ;
  printf("\n#\n") ;

  clean_memory() ;
  return 0 ;
}


void parse_arguments(int argc, char **argv)
{
  int c ;

  strcpy(cmd, argv[0]) ;

  T = 300.0 ;
  couple_min = 2.0 ;
  pHmin = -10.0 ;
  pHmax =  30.0 ;
  dpH   =   0.5 ;
  /* E is actually F*E, given in meV */
  Emin =  0.0 ;
  Emax =  0.0 ;
  dE   =  1.0 ;
  seed = 1234567 ;
  freeze_cutoff = 0.00 ;
  eqsteps = 1000 ;
  taumax = 20 ;
  nset = 0 ;
  compute_set = 0 ;
  compute_pair_corr = 0 ;
  compute_errors = 0 ;

  while ((c = getopt(argc, argv, "P:E:T:c:r:q:s:p:S:t:ed")) != -1)
  {
    switch(c)
    {
    case 'P':
      strsplit(optarg, ",") ;
      if (nsubstr != 3)	error('U', "Wrong argument for option -P.\n") ;
      pHmin = atof(substr[0]) ;
      pHmax = atof(substr[1]) ;
      dpH = atof(substr[2]) ;
      break ;
    case 'E':
      strsplit(optarg, ",") ;
      if (nsubstr != 3)	error('U', "Wrong argument for option -E.\n") ;
      Emin = atof(substr[0]) ;
      Emax = atof(substr[1]) ;
      dE = atof(substr[2]) ;
      break ;
    case 'T':
      T = atof(optarg) ;
      break ;
    case 'c':
      couple_min = atof(optarg) ;
      break ;
    case 'r':
      freeze_cutoff = atof(optarg) ;
      break ;
    case 'q':
      eqsteps = atol(optarg) ;
      break ;
    case 's':
      seed = atol(optarg) ;
      break ;
    case 'p':
      compute_pair_corr = 1 ;
      min_corr = atof(optarg) ;
      break ;
    case 'S':
      compute_set = 1 ;
      strcpy(sset, optarg) ;
      strsplit(sset, ",") ;
      if ((nset = nsubstr) < 2)
	error('U', "Option -S requires a minimum of 2 sites.\n") ;
      break ;
    case 't':
      compute_errors = 1 ;
      taumax = atoi(optarg) ;
      break ;
    case 'e':
      compute_energetics = 1 ;
      break ;
    case 'd':
      fprintf(stderr, "Version: %s\n", VERSION) ;
      fprintf(stderr, "Defaults:\n") ;
      fprintf(stderr, "  -P %f,%f,%f   -E %f,%f,%f\n",
	      pHmin, pHmax, dpH, Emin, Emax, dE) ;
      fprintf(stderr, "  -T %f   -c %f   -r %f   -q %ld   -s %ld\n",
	      T, couple_min, freeze_cutoff, eqsteps, seed) ;
      fprintf(stderr, "Option effects on calculation speed:\n") ;
      fprintf(stderr, "  -r : speeds up; suggested: -r 0.001 (negligible errors at titration endpoints)\n") ;
      fprintf(stderr, "  -S : load is negligible for sets up to ~5 elements, but becomes significant\n") ;
      fprintf(stderr, "       for sets above ~8, growing combinatorially.\n") ;
      fprintf(stderr, "  -p : slows down; suggested: -p 0.1\n") ;
      fprintf(stderr, "  -e : slows down\n") ;
      fprintf(stderr, "  -t : slows down a lot; suggested: -t 20\n") ;
      exit(0) ;
      break ;
    default:
      usage() ;
      exit(1) ;
    }
  }
  if (argc - optind != 3) error('U', "Wrong number of arguments\n") ;
  strcpy(pkint_file, argv[optind]) ;
  strcpy(inter_file, argv[optind+1]) ;
  mcsteps = atol(argv[optind+2]) ;
}


void usage(void)
{
  fprintf(stderr, "Usage: %s [options] pKint_file interactions_file MCsteps > ...\n", cmd) ;
  fprintf(stderr, "Options:\n") ;
  fprintf(stderr, "  -P pHmin,pHmax,dpH\t : solution pH range and increment\n") ;
  fprintf(stderr, "  -E Emin,Emax,dE   \t : solution potential range and increment (meV)\n") ;
  fprintf(stderr, "  -T temperature    \t : temperature (Kelvin) \n") ;
  fprintf(stderr, "  -c couple_min     \t : couple threshold for double flips (pK units) \n") ;
  fprintf(stderr, "  -r cutoff         \t : cutoff for reduced titration\n") ;
  fprintf(stderr, "                    \t   (full calculation if zero)\n") ;
  fprintf(stderr, "  -q eqsteps        \t : number of equilibration steps\n") ;
  fprintf(stderr, "  -s seed           \t : seed for random numbers\n") ;
  fprintf(stderr, "  -p min_corr       \t : cutoff for printing pair correlation coefficients\n") ;
  fprintf(stderr, "  -e                \t : compute energetics\n") ;
  fprintf(stderr, "  -S site1,site2,...\t : set of sites for microstate statistics\n") ;
  fprintf(stderr, "                    \t   (whose calculation is switched on)\n") ;
  fprintf(stderr, "  -t taumax         \t : maximum correlation time\n") ;
  fprintf(stderr, "                    \t   (switches on calculation of errors)\n") ;
  fprintf(stderr, "  -d                \t : shows defaults (if alone) and other info\n") ;
}


void make_initializations(void)
{
  int i ;
  char line[LINESIZE] ;
  FILE *fp ;

  pHsteps = rint(1 + (pHmax - pHmin) / dpH) ;
  Esteps  = rint(1 + (Emax - Emin) / dE) ;
  srand48(seed) ;
  srand(seed) ;

  if ((fp = fopen(pkint_file, "r")) == NULL)
    error('E', "Error opening file %s.\n", pkint_file) ;
  nsites = 0 ;
  while (fgets(line, LINESIZE-1, fp) != NULL)
  {
    line[strlen(line)-1] = '\0' ;
    strsplit(line, " ") ;
    if (nsubstr >= 3) nsites++ ;
  }
  fclose(fp) ;

  pkint = calloc(nsites, sizeof(float)) ;
  pkhalf = calloc(nsites, sizeof(float *)) ;
  for (i = 0 ; i < nsites ; i++)
    pkhalf[i] = calloc(MAXNPKHALFS, sizeof(float)) ;   
  npkhalfs = calloc(nsites, sizeof(int *)) ;
  min_inter = calloc(nsites, sizeof(float)) ;
  max_inter = calloc(nsites, sizeof(float)) ;
  titration = calloc(nsites, sizeof(int)) ;
  frozen = calloc(nsites, sizeof(int)) ;
  w = calloc(nsites, sizeof(float *)) ;
  for (i = 0 ; i < nsites ; i++) w[i] = calloc(nsites, sizeof(float)) ;
  site_name = (char **) calloc(nsites, sizeof(char *)) ;
  for (i = 0 ; i < nsites ; i++) site_name[i] = calloc(STRSIZE, sizeof(char)) ;
  site_type = calloc(nsites, sizeof(char)) ;
  z0 = calloc(nsites, sizeof(int)) ;
  z  = calloc(nsites, sizeof(int)) ;
  state = calloc(nsites, sizeof(int)) ;
  u = calloc(nsites, sizeof(float)) ;
  pair1 = calloc(nsites * (nsites - 1) / 2, sizeof(int)) ;
  pair2 = calloc(nsites * (nsites - 1) / 2, sizeof(int)) ;
  avg = calloc(nsites, sizeof(int)) ;
  pmean = calloc(nsites, sizeof(float)) ;
  binP = calloc(nsites + 1, sizeof(int)) ;
  binR = calloc(nsites + 1, sizeof(int)) ;
  if (compute_pair_corr)
  {
    avgpair = calloc(nsites, sizeof(int *)) ;
    for (i = 0 ; i < nsites ; i++) avgpair[i] = calloc(nsites, sizeof(int)) ;
  }
  if (compute_energetics) avgUn = calloc(nsites, sizeof(double)) ;
  if (compute_set)
  {
    setsite = calloc(nset, sizeof(int)) ;
    strsplit(sset, ",") ;
    for (i = 0 ; i < nset ; i++)
    {
      setsite[i] = atoi(substr[i]) ;
      if (setsite[i] < 0 || setsite[i] >= nsites)
	error('E', "Site number given to -S is out of range.\n") ;
    }
    nmicro = 1 << nset ;
    avgmicro = calloc(nmicro, sizeof(int)) ;
  }
  if (compute_errors)
  {
    cf =  calloc(nsites, sizeof(int *)) ;
    for (i = 0 ; i < nsites ; i++) cf[i] = calloc(taumax, sizeof(int)) ;
    cc = calloc(nsites, sizeof(int *)) ;
    for (i = 0 ; i < nsites ; i++) cc[i] = calloc(taumax, sizeof(int)) ;
    buf = calloc(nsites, sizeof(int *)) ;
    for (i = 0 ; i < nsites ; i++) buf[i] = calloc(taumax, sizeof(int)) ;
  }

  printf("# MCRP: Monte Carlo for Reduction and Protonation \n") ;
  printf("# version: %s \n#\n", VERSION) ;
  printf("# Input files: pKints = %s,  pairwise interactions = %s\n",
	 pkint_file, inter_file) ;
  printf("# Total number of titrable sites = %d\n", nsites) ;
  printf("# pH (min,max,delta): %f,%f,%f\n", pHmin, pHmax, dpH) ;
  printf("# E  (min,max,delta): %f,%f,%f\n", Emin, Emax, dE) ;
  printf("# Temperature = %f K\n", T) ;
  printf("# Production MC steps per (E,pH) point = %ld\n", mcsteps) ;
  printf("# Equilibration MC steps per (E,pH) point = %ld\n", eqsteps) ;
  printf("# Couple treshold for double state flips = %f pH units\n",
	 couple_min) ;
  if (freeze_cutoff != 0)
    printf("# Freeze cutoff for reduced titration = %f\n", freeze_cutoff) ;
  else printf("# Reduced titration switched off.\n") ;
  if (compute_pair_corr)
    printf("# Site-site correlations r computed and written if |r| >= %f.\n",
	   min_corr) ;
  else printf("# Computation of site-site correlations switched off.\n") ;
  if (compute_set)
  {
    printf("# Set of sites selected for microstate statistics: %d",
	   setsite[0]) ;
    for (i = 1 ; i < nset ; i++) printf(",%d", setsite[i]) ;
    printf("\n") ;
  }
  else printf("# Computation of microstate statistics switched off.\n") ;
  if (compute_errors)
    printf("# Maximum correlation time used for error calculation = %d\n",
	   taumax) ;
  else printf("# Error calculation switched off.\n") ;
  if (compute_energetics)
    printf("# Energetics calculations switched on.\n") ;
  else printf("# Energetics calculations switched off.\n") ;
  printf("# Seed for random number generator = %ld\n", seed) ;
  printf("#\n") ;
}


void read_pkint(void)
{
  int i ;
  char stit, line[LINESIZE] ;
  FILE *fp ;

  if ((fp = fopen(pkint_file, "r")) == NULL)
    error('E', "Error opening file %s.\n", pkint_file) ;
  for (i = 0 ; i < nsites ; i++)
  {
    fgets(line, LINESIZE-1, fp) ;
    line[strlen(line)-1] = '\0' ;
    strsplit(line, " ") ;
    if (nsubstr < 3)
      error('E', "Not enough fields in file '%s'.\n", pkint_file) ;
    pkint[i] = atof(substr[0]) ;
    switch(substr[1][0])
    {
    case 'A':  /* anionic */
      z0[i] = -1 ;
      break ;
    case 'C':  /* cationic */
      z0[i] = 0 ;
      break ;
    default:
      error('E', "Site %s has wrong charge type.\n", substr[2]) ;
    }
    strcpy(site_name[i], substr[2]) ;
    if (nsubstr >= 4) site_type[i] = substr[3][0] ;
    else site_type[i] = 'P' ;
    switch(site_type[i])
    {
    case 'P':  /* protonable */
      nsitesP++ ;
      break ;
    case 'R':  /* redox */
      nsitesR++ ;
      break ;
    default:
      error('E', "Site %s has wrong ligand.\n", substr[2]) ;
    }
    if (nsubstr >= 5) stit = substr[4][0] ;
    else stit = '*' ;
    switch(stit)
    {
    case '*':  /* titrable trough single and double flips */
      titration[i] = 3 ;
      state[i] = 0 ;
      break ;
    case 'd':  /* titrable through double flips only */
      titration[i] = 2 ;
      state[i] = 0 ;
      break ;
    case 's':  /* titrable through single flips only */
      titration[i] = 1 ;
      state[i] = 0 ;
      break ;
    case '0':  /* non-titrable, state 0 */
      titration[i] = 0 ;
      state[i] = 0 ;
      break ;
    case '1':  /* non-titrable, state 1 */
      titration[i] = 0 ;
      state[i] = 1 ;
      break ;
    default:
      error('E', "Site %s has wrong titration.\n", substr[2]) ;
    }
    z[i] = z0[i] + state[i] ;
  }
  fclose(fp) ;
}


void read_inter(void)
{
  int i, j ;
  float wtmp ;
  char line[LINESIZE] ;
  FILE *fp ;

  if ((fp = fopen(inter_file, "r")) == NULL)
    error('E', "Error opening file %s.\n", inter_file) ;
  for (i = 0 ; i < nsites ; i++)
  {
    for (j = 0 ; j < nsites ; j++)
    {
      fgets(line, LINESIZE-1, fp) ;
      line[strlen(line)-1] = '\0' ;
      strsplit(line, " ") ;
      if (nsubstr != 3)
	error('E', "Wrong number of fields in file '%s'.\n", inter_file) ;
      wtmp = atof(substr[2]) / (kBoltz_au * T) ;
      /* MAYBE CHANGE THIS FOR THE CASE OF NON-TITRABLE SITES..... */
      min_inter[i] += wtmp * z0[j] ;
      max_inter[i] += wtmp * (z0[j] + 1) ;
      w[i][j] = wtmp ;
    }
  }
  fclose(fp) ;
}


void select_pairs(void)
{
  int i, j ;

  printf("### Coupled pairs, with w >= %f pH units :\n",couple_min) ;
  npairs = 0 ;
  for (i = 0 ; i < nsites - 1 ; i++)
    if (titration[i] >= 2)
      for (j = i + 1 ; j < nsites ; j++)
	if (titration[j] >= 2)
	  if (w[i][j] >= LN10 * couple_min)
          {
	    pair1[npairs] = i ;
	    pair2[npairs] = j ;
	    printf("###   %3d %3d  %10.6f\n", i, j, w[i][j] / LN10) ;
	    npairs++ ;
	  }
  printf("### Total number of coupled pairs = %d\n#\n", npairs) ;
}


void initialize_EpH_point(float E, float pH)
{
  int n ;
  float pR, min_state, max_state ;

  /* - 2.3 kT pR <-> E (where E is really F*E) */
  pR = - E / (LN10 * kBoltz_meV * T) ;
  for (n = 0 ; n < nsites ; n++)
  {
    if (site_type[n] == 'R') u[n] = - LN10 * (pkint[n] - pR) ;
    else u[n] = - LN10 * (pkint[n] - pH) ;
    min_state = exp(-u[n] - max_inter[n]) / (1 + exp(-u[n] - max_inter[n])) ;
    max_state = exp(-u[n] - min_inter[n]) / (1 + exp(-u[n] - min_inter[n])) ;
    if (titration[n] >= 1)
    {
      if (freeze_cutoff != 0)
      {
	if (min_state > 1 - freeze_cutoff)
	{
	  state[n] = 1 ;
	  z[n] = z0[n] + 1 ;
	  frozen[n] = 1 ;
	  /* printf("> site %3d frozen in state 1\n", n) ; */
	}
	else if (max_state < freeze_cutoff)
	{
	  state[n] = 0 ;
	  z[n] = z0[n] ;
	  frozen[n] = 1 ;
	  /* printf("> site %3d frozen in state 0\n", n) ; */
	}
	else frozen[n] = 0 ;
      }
    }
  }
}


void initialize_E_line(float E)
{
  int n ;

  for (n = 0 ; n < nsites ; n++) npkhalfs[n] = 0 ;
}


void mc_step(float E, float pH)
{
  int i, j, k, p, di, dj ;
  float dU, sumi, sumj ;

  if (rand() / (RAND_MAX + 1.0) < 0.5)
  {
    for (i = 0 ; i < nsites ; i++)
    {
      if (titration[i] % 2 == 0) continue ;  /* if single flips not allowed */
      if (frozen[i]) continue ;
      di = 1 - 2 * state[i] ;
      dU = u[i] ;
      for (k = 0 ; k < nsites ; k++) dU += z[k] * w[i][k] ;
      dU *= di ;
      if (accept_move(dU))
      {
	state[i] += di ;
	z[i] += di ;
      }
    }
  }
  else
  {
    for (i = nsites - 1 ; i >= 0 ; i--)
    {
      if (titration[i] % 2 == 0) continue ;  /* if single flips not allowed */
      if (frozen[i]) continue ;
      di = 1 - 2 * state[i] ;
      dU = u[i] ;
      for (k = 0 ; k < nsites ; k++) dU += z[k] * w[i][k] ;
      dU *= di ;
      if (accept_move(dU))
      {
	state[i] += di ;
	z[i] += di ;
      }
    }
  }

  for (p = 0 ; p < npairs ; p++)
  {
    i = pair1[p] ;
    j = pair2[p] ;
    if (frozen[i] || frozen[j]) continue ;
    di = 1 - 2 * state[i] ;
    dj = 1 - 2 * state[j] ;
    dU = di * u[i] + dj * u[j] +
         ((z0[i] + 0.5) * dj + (z0[j] + 0.5) * di) * w[i][j] ;
    sumi = sumj = 0 ;
    for (k = 0 ; k < nsites ; k++)
      if (k != i && k != j)
      {
	sumi += z[k] * w[i][k] ;
	sumj += z[k] * w[j][k] ;
      }
    dU += di * sumi + dj * sumj ;
    if (accept_move(dU))
    {
      state[i] += di ;
      state[j] += dj ;
      z[i] += di ;
      z[j] += dj ;
    }
  }
}


int accept_move(float dU)
{
  if (dU > MAXDELTA) return 0 ;
  if (dU <= 0.0) return 1 ;
  if (invexp(dU) > drand48()) return 1 ;
  else return 0 ;
}


/* This function is a picewise rational polynomial approximation of
   exp(-x) in the intervals [0,6], [6,13] and [13,70]. Thus, it may
   give significant (even drastic) errors outside this range. */
float invexp(float x)
{
  float x2, x3, x4 ;

  if (x > 13)
  {
    return 8.194236147130614e-10 - 1.3290994520804703e-11 * x ;
  }
  else
  {
    x2 = x * x ;
    x3 = x2 * x ;
    if (x > 6)
    {
      return (-0.0013245823657199278 + 0.00027464252539452071 * x -
	      0.000019314947607346905 * x2 + 4.598224667374957e-7 * x3) /
	     (1 - 0.5165170691890946 * x + 0.09211442135429947 * x2 -
	      0.006143102546214945 * x3) ;
    }
    else
    {
      x4 = x2 * x2 ;
      return (0.9999965470613797 - 0.3960827416191208 * x +
	      0.06303500815508939 * x2 - 0.00476617578304489 * x3 +
	      0.00014392025197088043 * x4)/
	     (1 + 0.6038220689877429 * x + 0.16732494517488303 * x2 +
	      0.026354026827091058 * x3 + 0.00289071552898347 * x4) ;
    }
  }
}


void compute_statistics(int t)
{
  int i, statei, nP, nR ;
  int j, tt, tau0, tau ;
  float energy ;

  if (t == 0)
  {
    avgP2 = avgR2 = avgPR = 0 ;
    if (compute_energetics) avgU = avgU2 = 0.0 ;
    for (i = 0 ; i < nsites ; i++)
    {
      avg[i] = 0 ;
      if (compute_energetics) avgUn[i] = 0.0 ;
      if (compute_pair_corr)
	for (j = i + 1 ; j < nsites ; j++) avgpair[i][j] = 0 ;
      if (compute_errors)
	for (tau = 0 ; tau < taumax ; tau++)
	  cf[i][tau] = cc[i][tau] = buf[i][tau] = 0 ;
    }
    for (i = 0 ; i < nsites + 1 ; i++) binP[i] = binR[i] = 0 ;
    if (compute_set)
      for (j = 0 ; j < nmicro ; j++) avgmicro[j] = 0 ;
  }


  nR = nP = 0 ;
  energy = 0.0 ;
  for (i = 0 ; i < nsites ; i++)
  {
    avg[i] += statei = state[i] ;
    if (site_type[i] == 'P') nP += statei ;
    else nR += statei ;

    if (compute_pair_corr)
    {
      for (j = i + 1 ; j < nsites ; j++) avgpair[i][j] += statei * state[j] ;
    }

    if (compute_energetics)
    {
      energy += statei * u[i] ;
      for (j = i + 1 ; j < nsites ; j++) energy += z[i] * z[j] * w[i][j] ;
    }

    if (compute_errors)
    {
      tt = t % taumax ;
      buf[i][tt] = statei ;
      for (tau0 = 0 ; tau0 < taumax ; tau0++)
      {
	tau = t - tau0 ;
	if (tau < 0) continue ;
	tau = tau % taumax ;
	cf[i][tau] += buf[i][tau0] * statei ;
	cc[i][tau] += buf[i][tau0] + statei ;
      }
    }
  }
  avgP2 += nP * nP ;
  avgR2 += nR * nR ;
  avgPR += nP * nR ;
  binP[nP]++ ;
  binR[nR]++ ;

  if (compute_energetics)
  {
    /* offset energy to minimize roundoff errors */
    if (t == 0) energy0 = energy ;
    energy -= energy0 ;
    avgU += energy ;
    avgU2 += energy * energy ;
    for (i = 0 ; i < nsites ; i++) avgUn[i] += state[i] * energy ;
  }

  if (compute_set)
  {
    j = 0 ;
    for (i = 0 ; i < nset ; i++)
      if (state[setsite[i]]) j += 1 << i ;
    avgmicro[j]++ ;
  }

}


void write_EpH_point(float E, float pH)
{
  int i, j, k ;
  float meani, p, meanP = 0, meanR = 0, stdevP, stdevR, meanU,
        stdevU, dGi, dHi, TdSi, meanj, corr, pLi ;
  int tau, tcor ;
  float corf, corf0 = 0, err ;

  printf("#### E=%f   pH=%f\n", E, pH) ;

  if (compute_energetics) meanU = avgU / (float) mcsteps ;

  for (i = 0 ; i < nsites ; i++)
  {
    meani = avg[i] / (float) mcsteps ;
    if (site_type[i] == 'P' && pH != pHmin)
    {
      p = pmean[i] ;
      if (((p > 0.5) && (meani <= 0.5)) || ((p < 0.5) && (meani >= 0.5)))
      {
	if (npkhalfs[i] == MAXNPKHALFS)
	  error('W', "Number of pKhalf values exceeds %d for site %s.\n",
		MAXNPKHALFS, site_name[i]) ;
	else
	  pkhalf[i][npkhalfs[i]++] = pH - dpH * (meani - 0.5) / (meani - p) ;
      }
    }
    if (site_type[i] == 'P') meanP += meani ;
    else meanR += meani ;
    pmean[i] = meani ;
    printf(". %8.3f  %7.3f  %4d  %14.8e", E, pH, i, meani) ;

    if (compute_errors)
    {
      tcor = taumax ;
      for (tau = 0 ; tau < taumax ; tau++)
      {
	corf = (cf[i][tau] - meani * cc[i][tau]) / (float) (mcsteps - tau) +
	       meani * meani ;
	if (tau == 0) corf0 = corf ;
	if (corf0 == 0) corf = 1 ;
	else corf = corf / corf0 ;
	if ((tcor == taumax) && (fabs(corf) < 0.10)) tcor = tau ;
      }
      err = sqrt(corf0 * tcor / (float) mcsteps) ;
      printf("  %6.4f (%d)", err, tcor) ;
    }

    if (compute_energetics)
    {
      /* kT units */
      if (site_type[i] == 'P') pLi = pH ;
      else pLi = - E / (LN10 * kBoltz_meV * T) ;
      if (meani != 0 && meani != 1)
      {
	dGi = - log(meani / (1.0 - meani)) - LN10 * pLi ;
	TdSi = (avgUn[i] / (float) mcsteps - meani * meanU) / 
	       (meani - meani * meani) + log(meani / (1.0 - meani)) ;
	dHi = dGi + TdSi ;
	printf("  %13.6e %13.6e %13.6e", dGi, dHi, TdSi) ;
      }
      else printf("   -----------   -----------   ---------- ") ;
    }

    printf("\n") ;
  }
  stdevP = sqrtp(avgP2 / (float) mcsteps - meanP * meanP) ;
  stdevR = sqrtp(avgR2 / (float) mcsteps - meanR * meanR) ;
  printf(". %8.3f  %7.3f  totP  %14.8e %14.8e\n", E, pH, meanP, stdevP) ;
  printf(". %8.3f  %7.3f  totR  %14.8e %14.8e\n", E, pH, meanR, stdevR) ;

  printf("P %8.3f  %7.3f ", E, pH) ;
  for (i = 0 ; i < nsitesP + 1 ; i++) printf(" %d", binP[i]) ;
  printf("\n") ;
  printf("R %8.3f  %7.3f ", E, pH) ;
  for (i = 0 ; i < nsitesR + 1 ; i++) printf(" %d", binR[i]) ;
  printf("\n") ;

  if (compute_energetics)
  {
    stdevU = sqrtp(avgU2 / (float) mcsteps - meanU * meanU) ;
    /* energy offset has to be added to meanU but not to stdevU */
    printf("e %8.3f  %7.3f  %13.6e %13.6e\n",
	   E, pH, meanU + energy0, stdevU) ;
  }

  if (compute_pair_corr)
  {
    for (i = 0 ; i < nsites - 1 ; i++)
    {
      meani = avg[i] / (float) mcsteps ;
      for (j = i + 1 ; j < nsites ; j++)
      {
	meanj = avg[j] / (float) mcsteps ;
	if (meani == 0 || meani == 1 || meanj == 0 || meanj == 1)
	  corr = 0 ;
	else 
	  corr = (avgpair[i][j] / (float) mcsteps - meani * meanj) /
	         sqrtp((meani - meani * meani) * (meanj - meanj * meanj)) ;
	if (fabs(corr) >= min_corr)
	  printf(": %8.3f  %7.3f  %4d %4d  %13.6e  %13.6e\n",
		 E, pH, i, j, corr, avgpair[i][j] / (float) mcsteps) ;
      }
    }
  }
  if (stdevP == 0 || stdevR == 0) corr = 0 ;
  else corr = (avgPR / (float) mcsteps - meanP * meanR) / (stdevP * stdevR) ;
  printf(": %8.3f  %7.3f  totP totR  %13.6e  %13.6e\n",
	 E, pH, corr, avgPR / (float) mcsteps) ;

  if (compute_set)
  {
    for (j = 0 ; j < nmicro ; j++)
    {
      printf("m %8.3f  %7.3f  ", E, pH) ;
      for (i = 0, k = j ; i < nset ; i++) printf("%1d", 1 & (j >> i)) ;
      printf("  %14.8e\n", avgmicro[j] / (float) mcsteps) ;
    }
  }

}


void write_E_line(float E)
{
  int n, i ;

  printf("#\n# pKhalfs for E = %f :\n", E) ;
  printf("#   site       number     E       pKint    pKhalf(s)\n") ;
  printf("#---------------------------------------------------------\n") ;
  for (n = 0 ; n < nsites ; n++)
  {
    if (titration[n] == 0)
      printf("> %-13s %4d %8.3f %9.4f  NonTitrable\n",
	     site_name[n], n, E, pkint[n]) ;
    else if (site_type[n] == 'R')
      printf("> %-13s %4d %8.3f %9.4f      ----\n",
	     site_name[n], n, E, pkint[n]) ;
    else if (npkhalfs[n] == 0)
      printf("> %-13s %4d %8.3f %9.4f  NotInRange\n",
	     site_name[n], n, E, pkint[n]) ;
    else
    {
      printf("> %-13s %4d %8.3f %9.4f  %9.4f",
	     site_name[n], n, E, pkint[n], pkhalf[n][0]) ;
      for (i = 1 ; i < npkhalfs[n] ; i++) printf(", %.4f", pkhalf[n][i]) ;
      printf("\n") ;
    }
  }
  printf("#\n") ;
}


void strsplit(char *s, const char *delim)
{
  int i ;
  char *c, aux[LINESIZE] ;

  if (substr != NULL)
  {
    for (i = 0 ; i < nsubstr ; i++) free(substr[i]) ;
    free(substr) ;
  }
  strcpy(aux, s) ;
  c = strtok(aux, delim) ;
  for (nsubstr = 0 ; c != NULL ; nsubstr++) c = strtok(NULL, delim) ;
  if (nsubstr > 0)
  {
    substr = (char **) calloc(nsubstr, sizeof(char *)) ;
    for (i = 0 ; i < nsubstr ; i++) substr[i] = calloc(STRSIZE, sizeof(char)) ;
    strcpy(aux, s) ;
    strcpy(substr[0], strtok(aux, delim)) ;
    for (i = 1 ; i < nsubstr ; i++) strcpy(substr[i], strtok(NULL, delim)) ;
  }
}


double sqrtp(double x)
{
  return (x>0 ? sqrt(x) : 0) ;
}


void clean_memory(void)
{
  int i ;

  free(pkint) ;
  for (i = 0 ; i < nsites ; i++) free(pkhalf[i]) ;
  free(pkhalf) ;
  free(npkhalfs) ;
  free(min_inter) ;
  free(max_inter) ;
  free(titration) ;
  free(frozen) ;
  for (i = 0 ; i < nsites ; i++) free(w[i]) ;
  free(w) ;
  for (i = 0 ; i < nsites ; i++) free(site_name[i]) ;
  free(site_name) ;
  free(site_type) ;
  free(z0) ;
  free(z) ;
  free(state) ;
  free(u) ;
  free(pair1) ;
  free(pair2) ;
  free(avg) ;
  free(pmean) ;
  if (compute_pair_corr)
  {
    for (i = 0 ; i < nsites ; i++) free(avgpair[i]) ;
    free(avgpair) ;
  }
  if (compute_energetics) free(avgUn) ;
  if (compute_set)
  {
    free(setsite) ;
    free(avgmicro) ;
  }
  if (compute_errors)
  {
    for (i = 0 ; i < nsites ; i++) free(cf[i]) ;
    free(cf) ;
    for (i = 0 ; i < nsites ; i++) free(cc[i]) ;
    free(cc) ;
    for (i = 0 ; i < nsites ; i++) free(buf[i]) ;
    free(buf) ;
  }
  if (substr != NULL)
  {
    for (i = 0 ; i < nsubstr ; i++) free(substr[i]) ;
    free(substr) ;
  }
}


void error(char errtype, char *format, ...)
{
  va_list args;

  va_start(args, format);
  if (errtype != 'W' && errtype != 'E' && errtype != 'U')
    error('E', "Wrong use of error function.\n") ;
  if (errtype == 'W') fprintf(stderr, "%s: WARNING: ", cmd);
  else fprintf(stderr, "%s: ERROR: ", cmd);
  vfprintf(stderr, format, args);
  va_end(args);
  if (errtype == 'U') usage() ;
  if (errtype != 'W') exit(1) ;
  return ;
}

