/*
This file is part of meadTools, version 2.2.

Copyright (c) 2001-2019, Instituto de Tecnologia Quimica e Biologica,
Universidade Nova de Lisboa, Portugal.

meadTools is free software: you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation, either version 2 of the License, or (at your
option) any later version.

meadTools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with meadTools.  If not, see <http://www.gnu.org/licenses/>.

For further details and info check the README file.

You can get meadTools at www.itqb.unl.pt/simulation
*/


/************************************************************************/
/* cconvert:                                                            */
/*                                                                      */
/* This is a program to convert meadT output into petit input.          */
/* It is a direct translation of the original awk version,              */
/* which should be checked for further info.                            */
/*                                                                      */
/* Carlos A. Cunha, ITQB-UNL, Oeiras, Março 2005                        */
/*                                                                      */
/* Acknowledgment: financial support from Fundacao para a Ciencia e a   */
/*                 Tecnologia, Portugal (grant SFRH/BPD/9466/2002).     */
/*                                                                      */
/************************************************************************/

/*Program exit status was modified (returns 0) and now can be used*/
/*inside bash scripts without killing them; scampos 16-11-2011*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
/* #include <math.h> */

#define MXLNLGTH  256  /* Maximum length for strings (lines) read from files. */
#define MAXNFLDS   10  /* Maximum number of fields a line read from file can contain */
#define MAXFLDSIZE 20  /* Maximum number of characters a field can contain, */
                       /* counting also the null character at the end.      */

#define MAXPSEUDOS 10000 /* Maximum number of pseudo-sites. */
#define MAXNSITES   5000 /* Maximum number of sites.        */
#define MAXNTAUTS     10 /* Maximum number of tautomers per site. */


int main(int argc, char *argv[]) { /*scampos 16-11-2011*/

 const double  kBoltz_au  = 5.98435e-6 ;  /*  e^2/(Angstrom*K)  */
 /* const double  kBoltz_meV = 0.0861734 ; */  /*  meV/K  */

 /* const double LN10 = log(10.0); */
 const double LN10 = 2.302585093;
 
 FILE *pkcrg_file, *g_file, *dat_file;

 char *pkcrg_name, *g_name, *dat_name;

 char line[MXLNLGTH];

 char flds[MAXNFLDS][MAXFLDSIZE];

 static char site_name[MAXPSEUDOS][MAXFLDSIZE];
 static char site_type[MAXPSEUDOS][2];

 char aux[MAXFLDSIZE];
 char stit[2];

 char tit[5][2] = {
                   {'f','\0'}, 
                   {'s','\0'},
                   {'d','\0'},
                   {'*','\0'},
		   {'n','\0'}
                           };

 static double pkcrg[MAXPSEUDOS];
 static double w[MAXPSEUDOS][MAXPSEUDOS];

 double T, g, gg;

 static int z0[MAXPSEUDOS];
 static int titration[MAXPSEUDOS];
 static int state[MAXPSEUDOS];            
 static int ntauts[MAXNSITES];
 static int nstates[MAXNSITES];
 static int taut[MAXNSITES][MAXNTAUTS];
 static int stateocc[MAXNSITES][MAXNTAUTS];

 int npseudos, nsites, NF;

 int i, j, si, sj, k, t, pstn, dpstn, nchrs;


  /* Function templates */

 int csplit(char *, char *, int, int, char *);

 int findsubstr(char *, char *);

 int findigits(char *, int);

 int delsubstr(char *, int, int);



  /* Getting the command line parameters... */

  /* Modified to make temperature specification non-optional (AMB) */
 if (argc != 5) {
    printf("\nUsage:  cconvert  pkcrg_file  g_file  temperature  dat_file\n\n") ;
    exit(1);
 }

 pkcrg_name = argv[1];
 g_name = argv[2];
 T = atof(argv[3]);   /* testar */
 dat_name = argv[4] ;


  /*   Reading file .pkcrg   */


 if ( (pkcrg_file=fopen(pkcrg_name, "r")) == NULL ) {        
   printf("\n\n\n     ERROR: Cannot open .pkcrg file %s !\n\n\n", pkcrg_name);
   exit(2);
 }
 else printf("\ncconvert: file %s opened for reading.\n", pkcrg_name);

 printf("\ncconvert: counting npseudos in .pkcrg file...\n");

 npseudos = 0;

 while(fgets(line, MXLNLGTH, pkcrg_file)!=NULL) {

   NF = csplit(line," ", MAXNFLDS, MAXFLDSIZE, flds[0]);

   if (NF >= 3) npseudos++ ;

 }

 if (npseudos > MAXPSEUDOS) {

   printf("\n\ncconvert ERROR: number of pseudo-sites exceeds MAXPSEUDOS!");
   printf("\n                Arrays size exceeded. Increase MAXPSEUDOS!");
   printf("\nExiting...\n\n");

   exit(3);
 }

 rewind(pkcrg_file) ;

 printf("\ncconvert: reading npseudos from .pkcrg file...\n");

 for (i = 0 ; i < npseudos ; i++) {

   if (fgets(line, MXLNLGTH, pkcrg_file)!=NULL) {

     NF = csplit(line," ", MAXNFLDS, MAXFLDSIZE, flds[0]);

     if (NF < 3) printf("\ncconvert ERROR: not enough fields in file %s !\n", pkcrg_name) ;

     pkcrg[i] = atof(flds[0]);   /* testar */

     /* Of the following fields only the ones referring to the first */
     /* tautomer are relevant for the calculations. */

     if (!strcmp(flds[1],"A")) z0[i] = -1 ;      /* anionic  */     /* testar */
     else if (!strcmp(flds[1],"C")) z0[i] = 0 ;  /* cationic */
     else printf("\ncconvert ERROR: pseudo-site %s has wrong charge type!\n", flds[2]) ;

     strcpy(site_name[i], flds[2]);   /* testar */

     if (NF >= 4) strcpy(site_type[i], flds[3]);
     else  strcpy(site_type[i], "P");

     if (strcmp(site_type[i],"P") && strcmp(site_type[i],"R")) {
       printf("\ncconvert ERROR: pseudo-site %s has wrong ligand type!\n", flds[2]) ;
     }

     if (NF >= 5) strcpy(stit, flds[4]);
     else if (pkcrg[i] < -70)  strcpy(stit,"n");
     else  strcpy(stit,"*");

     if (!strcmp(stit,"*")) titration[i] = 3 ;       /* titrable through single and double flips */
     else if (!strcmp(stit,"d")) titration[i] = 2 ;  /* titrable through double flips only */
     else if (!strcmp(stit,"s")) titration[i] = 1 ;  /* titrable through single flips only */
     else if (!strcmp(stit,"n")) titration[i] = 4 ;  /* neutral non-titrable but tautomerizable */
     else if (!strcmp(stit,"f")) {                   /* fixed state (only needed in the first pseudo-site) */
       titration[i] = 0 ;
       state[i] = atoi(flds[5]);      /* testar */
     }
     else printf("\ncconvert ERROR: pseudo-site %s has wrong titration!\n", flds[2]) ;
   }
 }

 fclose(pkcrg_file) ;


  /*   Reading interactions file (.g)   */


  if ( (g_file=fopen(g_name, "r")) == NULL ) {        
    printf("\n\n\n     ERROR: Cannot open .g file %s !\n\n\n", g_name);
    exit(2);
  }
  else printf("\ncconvert: file %s opened for reading.\n", g_name);

  printf("\ncconvert: reading interactions (.g) file...\n");

  for (i = 0 ; i < npseudos ; i++)
  for (j = 0 ; j < npseudos ; j++) {

   if (fgets(line, MXLNLGTH, g_file)!=NULL) {

     NF = csplit(line," ", MAXNFLDS, MAXFLDSIZE, flds[0]);

     if (NF != 3) printf("\ncconvert ERROR: wrong number of fields in file %s !\n", g_name) ;

     w[i][j] = atof(flds[2]);   /* testar */
   }
  }

  fclose(g_file) ;

  /* ATTENTION! The following assumes that pseudo-sites of the same site */
  /* are contiguous in the .g file. */

  t = 0;
  k = 0;
  for (i = 0 ; i < npseudos ; i += t)
  {
    t = 1 ;
    for (j = i + 1 ; j < npseudos ; j++)
    {
      if (w[i][j] > 0.5) t++ ;
      else break ;
    }
    ntauts[k++] = t ;
  }
  nsites = k ;

 if (nsites > MAXNSITES) {

   printf("\n\ncconvert ERROR:   number of sites exceeds MAXNSITES!");
   printf("\n                Arrays size exceeded. Increase MAXNSITES!");
   printf("\nExiting...\n\n");

   exit(3);
 }


  /*   Making the initializations...   */

  printf("\ncconvert: making the initializations...\n");

  j = 0 ;
  for (i = 0 ; i < nsites ; i++)
    for (t = 0 ; t < ntauts[i] ; t++) taut[i][t] = j++ ;

  /* The same-variable assignments in the following loop are ok */
  /* because we have always i <= taut[i,0], and therefore data needed */
  /* to the next assignments is never overwritten. */

  for (i = 0 ; i < nsites ; i++)
  {

    /* Tautomer-part of name must be removed. The procedure used here */
    /* may not be the most general one. */

    NF = csplit(site_name[taut[i][0]],"-", MAXNFLDS, MAXFLDSIZE, flds[0]);  

    /* flds = aux in the original awk version */

    /* gsub(/[a-z0-9]/, "", aux[1]) ; */
    /* Changed to use POSIX character classes: */
    /* gsub(/[[:lower:][:digit:]]/, "", aux[1]) ; */

    /* sub(/(all|avx|tau[[:digit:]]+)/, "", aux[1]) ; */

    strcpy(aux,flds[0]);

    if ((pstn = findsubstr(aux,"all")) > 0) delsubstr(aux, pstn, 3);   /* testar */
    else if ((pstn = findsubstr(aux,"avx")) > 0) delsubstr(aux, pstn, 3);
    else if ((pstn = findsubstr(aux,"tau")) > 0) {                     /* testar */
      dpstn = pstn + 3;
      nchrs = 3 + findigits(aux, dpstn);
      delsubstr(aux, pstn, nchrs);
    }

    strcat(aux,"-");
    strcat(aux,flds[1]);
    strcpy(site_name[i],aux);

    strcpy(site_type[i], site_type[taut[i][0]]);

    z0[i] = z0[taut[i][0]] ;
    titration[i] = titration[taut[i][0]] ;
    if (titration[i] == 4) nstates[i] = ntauts[i] ;
    else nstates[i] = ntauts[i] + 1 ;
    if (titration[i] == 0) state[i] = state[taut[i][0]] ;

    /* ie, only the first tautomer must be fixed in the .pkcrg file. */

    else state[i] = 0 ;
  }


  /*   Making the new matrix...   */

  printf("\ncconvert: making the new matrix...\n");

   if ( (dat_file=fopen(dat_name, "w")) == NULL ) {        
     printf("\n\n\n     ERROR: Cannot open .dat file %s !\n\n\n", dat_name);
     exit(2);
   }
   else printf("\ncconvert: file %s opened for writing.\n", dat_name);

   fprintf(dat_file, "%d\n", nsites);

  /* Compute occupancies of site states */

  for (i = 0 ; i < nsites ; i++)
  {
    for (si = 0 ; si < nstates[i] ; si++)
    {
      if (si == ntauts[i]) stateocc[i][si] = z0[i] + 1 ;
      else stateocc[i][si] = -z0[i] ;
    }
  }

  /* Compute individual terms */

  for (i = 0 ; i < nsites ; i++)
  {
    fprintf(dat_file, "%-10s %2d  %s  %s\n", site_name[i], nstates[i], site_type[i], tit[titration[i]]);
    for (si = 0 ; si < nstates[i] ; si++)
    {
      if (si == ntauts[i]) g = 0 ;
      else g = (2 * z0[i] + 1) * LN10 * (kBoltz_au * T) * pkcrg[taut[i][si]] ;
      fprintf(dat_file, "%1d %13.6e \n", stateocc[i][si], g);
    }
  }

  /* Compute interaction matrix */

  for (i = 0 ; i < nsites - 1 ; i++)
  for (j = i + 1 ; j < nsites ; j++)
  {
    for (si = 0 ; si < nstates[i] ; si++)
    for (sj = 0 ; sj < nstates[j] ; sj++)
    {
      if (si == ntauts[i] || sj == ntauts[j])
        gg = 0.0 ;
      else
        gg = (2 * z0[i] + 1) * (2 * z0[j] + 1) * w[taut[i][si]][taut[j][sj]] ;
      fprintf(dat_file, "%3d %2d   %3d %2d   %13.6e\n", i, si, j, sj, gg);
    }
  }

  fclose(dat_file) ;

  printf("\n\ncconvert WARNING: Make sure the redox sites are properly marked in the .pkcrg file!\n");

  printf("\n   ***  cconvert normal termination  ***\n");
  return 0 ; /*scampos 16-11-2011*/
}
