#!/bin/bash -e
# Adapt to your own gromacs' version and location
source /gromacs/gromacs-2020.5/bin/GMXRC

# We create a CG topology and a CG structure from the used pdb (see flag -h for usage):
martinize2 -f 4c69.pdb -x VDAC1_cg.pdb -o topol.top -ff martini3001 -elastic -scfix -cys auto -dssp dssp -ignore HOH ATP MC3 LDA -maxwarn 41

# We then use insane to create a 12x12 nm membrane, to embed the CG protein in it, and to addsolvent and ionic strength:
./insane.py -f VDAC1_cg.pdb -l POPC:95 -l DPCE:5 -x 12 -y 12 -z 10 -o membrane_prot.gro -p membrane.top -center -sol W -salt 0.15 -orient -charge auto

# We take the lipid/solvent counts from the .top created by insane:
tail -n 7 membrane.top >> topol.top

# We clean the .itp and the .top, and add the appropriate includes.
mv molecule_0.itp VDAC1.itp
sed -i 's/molecule_0/VDAC1/' VDAC1.itp
sed -i -e 's/molecule_0/VDAC1/' -e 's/martini.itp/martini_v3.0.0.itp/' topol.top
sed -i '4i\#include "martini_v3.0.0_solvents_v1.itp"\n#include "martini_v3.0.0_ions_v1.itp"\n#include "martini_v3.0.0_phospholipids_v1.itp"\n#include "DPCE.itp"' topol.top
sed -i -e 's/NA+/NA/' -e 's/CL-/CL/' topol.top

# We can now do an energy minimization
gmx grompp -f em.mdp -p topol.top -c membrane_prot.gro -o em.tpr -maxwarn 1
gmx mdrun -deffnm em -v

# We now need to do the index for the Solvent / Bilayer, for separate thermostating (we also add the backbone bead selection for later)
gmx make_ndx -f em.tpr <<EOF
"W" | "ION"
name 17 Solvent
! 17
name 18 Bilayer
a BB
q
EOF

# We equilibrate temperature and pressure
gmx grompp -f eq.mdp -p topol.top -c em.gro -n index.ndx -o eq.tpr -maxwarn 1
gmx mdrun -deffnm eq -v

# and do a production run 
gmx grompp -f md.mdp -p topol.top -c eq.gro -n index.ndx -o md.tpr
gmx mdrun -deffnm md -v


# Trajectory centering and fitting
echo Protein Bilayer | gmx trjconv -f md.xtc -s md.tpr -center -pbc mol -o md_pbc.xtc -n index.ndx 
echo Protein Bilayer | gmx trjconv -f eq.gro -s md.tpr -center -pbc mol -o md_pbc.gro -n index.ndx 
echo Bilayer | gmx convert-tpr -s md.tpr -n index.ndx -o fit.tpr -nsteps -1
echo BB Bilayer | gmx trjconv -f md_pbc.xtc -s fit.tpr -fit rotxy+transxy -o md_fit.xtc -n index.ndx 


# We get the contact distances and respective residue-averaged plot
echo Protein DPCE | gmx mindist -respertime -s fit.tpr -f md_fit.xtc -or res_dists.xvg
python3 contact_fraction.py

